-- Generierung Bestellnummer, Bestellnummer zusammenfassen
 -- Gibt in Abhängigkeit von Bestell-Code und Auftragsnummer eine neue Bestellnummer zurück
   -- DROP FUNCTION IF EXISTS TEinkauf.Generate_Bestellnummer(varchar,integer, boolean);
CREATE OR REPLACE FUNCTION TEinkauf.Generate_Bestellnummer(
      ldcode varchar,
      agid integer = null,
      previewnummer boolean = false
  )
  RETURNS varchar AS $$
  BEGIN
      RETURN (TEinkauf.Generate_Bestellnummer_Ext(ldcode, agid, previewnummer)).Nummer;
  END $$ LANGUAGE plpgsql STABLE;
 --

 -- Gibt in Abhängigkeit von Bestell-Code und Bestellvorschlags-PositionsID eine neue Bestellnummer zurück
CREATE OR REPLACE FUNCTION TEinkauf.Bestvorschlag_Generate_Bestellnummer(
      ldcode varchar,
      bvpid integer,
      previewnummer boolean = false,
      OUT nummer varchar,
      OUT numCircle varchar
  )
  RETURNS record AS $$
  DECLARE
      agid integer;
      r record;
  BEGIN
      -- Erste AG-ID der Artikelbedarfe ermitteln.
      SELECT la_ag_id, la_bap_id
        INTO r
        FROM ldsauftg
       WHERE la_bvp_id = bvpid
       ORDER BY la_id
       LIMIT 1
      ;

      -- Auf Basis der AG-ID und aktuellen Einstellungen die Bestellnummer holen
      -- EinkaufNummer = VerkaufNummer
      -- Keine AG-ID gefunden und kein sonstiger Bedarf, gibt normale/freie Bestellnummer zurück. => FBE
      -- Keine AG-ID aber BANF gefunden MIT Angebot/Auftrag gibt gebundene, fortlaufende Bestellnummer zurück => -1 https://redmine.prodat-sql.de/issues/12007
      agid := r.la_ag_id;
      --
      IF (
              FOUND
          AND agid IS NULL
          AND EXISTS( SELECT true FROM bestanfpos WHERE bap_id = r.la_bap_id AND bap_agnr IS NOT NULL )
      ) THEN
          -- Kennzeichen, das dennoch BE-Nummer (Gebundene BANF). siehe oben.
          agid := -1;
      END IF;
      --
      SELECT bNr.nummer, bnr.numCircle
        INTO nummer, numcircle
        FROM TEinkauf.Generate_Bestellnummer_Ext(ldcode, agid, previewnummer) AS bNr;
      --
      RETURN;
  END $$ LANGUAGE plpgsql STABLE;
   --

 -- Gibt in Abhängigkeit von Bestell-Code und Auftragsnummer eine neue Bestellnummer zurück
   -- DROP FUNCTION IF EXISTS TEinkauf.Generate_Bestellnummer_Ext(varchar,integer, boolean);
CREATE OR REPLACE FUNCTION TEinkauf.Generate_Bestellnummer_Ext(
    IN  _ldcode         varchar,
    IN  _agid           integer = null,
    IN  _previewnummer  boolean = false,
    OUT nummer          varchar,
    OUT numCircle       varchar
    )
    RETURNS record
    AS $$
    DECLARE _r record;
    BEGIN

      _agid := nullif( _agid, 0 );

      SELECT ag_astat, ag_nr, ag_pos
        INTO _r
        FROM auftg
       WHERE ag_id = _agid;
      --

      -- agid => -1 : es ist kein direkter Auftragsverursacher vorhanden, aber anderer Bedarfsverursacher (BANF). Daher wird es keine FBE sonder BE
      -- Auftragsnummer = Bestellnummer ist nicht möglich bei agid -1
      -- Siehe Bestvorschlag_Generate_Bestellnummer
      SELECT *
        INTO nummer, numCircle
        FROM TEinkauf.Generate_Bestellnummer_Ext_by_agnr(
                 _ldcode,
                 _r.ag_astat,
                 ifthen(
                     _agid = -1,
                     '-1',
                     _r.ag_nr
                 ),
                 _r.ag_pos,
                 _previewnummer
             );
      --
      RETURN;

    END $$ LANGUAGE plpgsql;
 --

CREATE OR REPLACE FUNCTION TEinkauf.anfart__alternative__by__aart_id__get(
                                      IN _aart_id             integer,
                                      OUT has_alt             boolean,
                                      OUT has_alt_aart_pos    varchar,
                                      OUT is_alt              boolean,
                                      OUT is_alt_zu_aart_id   integer,
                                      OUT is_alt_zu_aart_pos  integer
                                      )
  RETURNS record AS $$
DECLARE
  anfNr  varchar = (SELECT aart_anf_nr FROM anfart WHERE aart_id = _aart_id);
BEGIN
  --Datensatz hat Alternativposition(en)
  has_alt_aart_pos := (SELECT string_agg( aart_pos, ', ' ORDER BY aart_pos ASC )
                         FROM anfart
                        WHERE aart_anf_nr = anfNr
                          AND _aart_id = ANY ( aart_alternativ_zu_aart_id )
                      );

  has_alt := has_alt_aart_pos IS NOT NULL;

  --Datensatz ist eine Alternativposition zu anderem
  SELECT anfart_alternativen.aart_id, anfart_alternativen.aart_pos
    INTO is_alt_zu_aart_id, is_alt_zu_aart_pos
    FROM anfart
    JOIN anfart AS anfart_alternativen ON anfart_alternativen.aart_id = ANY (anfart.aart_alternativ_zu_aart_id)
   WHERE anfart.aart_id = _aart_id
   LIMIT 1;

  is_alt := is_alt_zu_aart_id IS NOT NULL;

  RETURN;
END $$ LANGUAGE plpgsql STABLE;

-- Hinzufügen einer Alternativposition zu einem Lieferantenangebot ~> Funktion für RTF #2020
 -- Fuunktion legt neuen Anfrageartikel in der bestehenden Lieferantenanfrage an.
 -- Dieser wird mit Lieferant verknüpft, als Alternativposition gekennzeichnet und als Lieferantenangebotsposition übernommen
CREATE OR REPLACE FUNCTION TEinkauf.anfangebot__alternative__add(
  _ak_nr      character varying,
  _ak_bez     character varying,
  _alief_lkn  character varying,
  _menge      numeric,
  _me_iso     character varying,
  _anf_nr     character varying,
  _aart_id    integer)
  RETURNS void AS $function$
DECLARE
  aartid      integer;
  mid         integer;
  aangid      integer;
  aliefid     integer;
BEGIN

  --get mg_code by ak_nr und me_iso
  SELECT
    m_mgcode
  FROM
    artmgc
  JOIN
    mgcode ON m_mgcode = me_cod
  WHERE
        m_ak_nr = _ak_nr
    AND me_iso = _me_iso
  INTO mid;

  IF mid IS NULL THEN
    mid = 1; --per Standard 'Stück' (idR für Artikel die nicht in 'art' Tabelle sind)
  END IF;

  --raise notice 'm_id: %', mid;

  --get alief_id by _alief_lkn
  SELECT
    alief_id
  FROM
    anflief
  WHERE
        alief_lkn = _alief_lkn
    AND alief_anf_nr = _anf_nr
  INTO aliefid;

  --raise notice 'alief_id: %', aliefid;

  -- Alternative als Anfrageartikel  hinzufügen
  INSERT INTO anfart (aart_pos,
                      aart_ak_nr,
                      aart_bez,
                      aart_anf_nr,
                      aart_menge,
                      aart_m_id,
                      aart_alief_id,
                      aart_alternativ_zu_aart_id
                     )
              VALUES ((SELECT MAX(aart_pos) + 1 FROM anfart WHERE aart_anf_nr = _anf_nr),
                      _ak_nr,
                      _ak_bez,
                      _anf_nr,
                      _menge,
                      mid,
                      ARRAY[aliefid], --und sofort aktuellem Lieferant anfragen
                      ARRAY[_aart_id] --Als Alternativartikel verlinken
                     )
     RETURNING aart_id INTO aartid;

  --raise notice 'aart_id: %', aartid;

  --Angebotene Alternative hinzufügen
  INSERT INTO anfangebot (aang_aLief_id,
                          aang_aart_id,
                          aang_menge
                         )
                  VALUES (aliefid,
                          aartid,
                          _menge)
  RETURNING aang_id INTO aangid;

  --raise notice 'aang_id: %', aangid;

  RETURN;
END $function$ LANGUAGE plpgsql;

 --
CREATE OR REPLACE FUNCTION TEinkauf.Generate_Bestellnummer_Ext_by_agnr(
    IN  ldcode        varchar,
    IN  astat         varchar = null,
    IN  agnr          varchar = null,
    IN  agpos         integer = null,
    IN  previewnummer boolean = false,
    OUT nummer        varchar,
    OUT numCircle     varchar
    )
    RETURNS record
    AS $$
    DECLARE
        rec record;
    BEGIN
      -- Erläuterung:
      --  Systemeinstellung Bestellnummer = Auftragsnummer?
      --  Ja
      --      Auftragsnummer beginnt mit 'AG' (ODER ag_astat='I' UND) ag_nr='LA' = freier Produktionsauftrag?
      --  Ja
      --      Code = 'I' => PA + Nummer       -- AG2014-1234 => 'PA2014-1234' = Produktionsauftrag
      --      Code = 'E' => BE + Nummer       -- AG2014-1234 => 'BE2014-1234' = Bestellung
      --      Code = 'R' => Nicht definiert.  -- ?
      --  Nein
      --      Code = 'I' => Auftragsnummer                -- 2014-1234
      --      Code = 'E' => Auftragsnummer                -- 2014-1234
      --      Code = 'R' => Lager-Produktionsauftrag      -- LA
      --  Nein (oder keine Auftragsnummer angegeben oder Status ='R'):
      --      Code = 'R' => Nummernkreis 'ldsdokr'    -- RBE2014-0012 = Rahmenbestellung
      --      Code = 'E' => Nummernkreis 'ldsdok'     -- FBE2014-1234 = Freie Bestellung
      --      Code = 'I' => Nummernkreis 'ldsdoki'    -- LA2014-1234  = Lagerauftrag
      --      (Wird der Nummernkreis 'ldsdoki' nicht gefunden, wird auf Nummernkreis 'ldsdok' zurückgegangen)

       -- Fehlermeldung keine Nummer
       IF ldcode IS NULL THEN
          RAISE EXCEPTION 'TEinkauf.Generate_Bestellnummer: %', format(lang_text(29174) );
       END IF;

       Nummer := null;
       NumCircle := null;

       -- Bestellnummer soll laut Systemeinstellung = Auftragsnummer sein
       IF     ( TSystem.Settings__Getbool('ld_auftg=ag_nr')
                OR
                (     TSystem.Settings__Getbool('ld_auftg=ag_nr_add_pos')
                  AND (   (astat = 'E' AND ldcode = 'I')
                       OR (astat = 'I' AND ldcode = 'I')
                      )
                )
              )
          AND -- AG => PA umgewandelt usw.
              -- Auftragsnummer -1 : fortlaufende BE-Nummer
              -- Keine AG-ID aber BANF gefunden MIT Angebot/Auftrag gibt gebundene, fortlaufende Bestellnummer zurück => -1 https://redmine.prodat-sql.de/issues/12007
              -- Siehe Bestvorschlag_Generate_Bestellnummer
              (nullIf(agnr, '-1') IS NOT NULL)
          AND  --Rahmen ausschliessen, da sonst evtl überschneidung mit AG-Nummern. Rahmen werden wie Lageraufträge angesehen und sind somit perse frei
              (ldcode <> 'R')
          AND (agnr NOT LIKE 'RA%')
       THEN
           -- Beginnt nicht mit 'AG...' oder 'PA....' (interner Bedarf) BE%: Folge-ABK F-ABK >> umbauen auf Status
           IF     (agnr NOT LIKE 'AG%')
              AND (agnr NOT LIKE 'PA%')
              AND (agnr NOT LIKE 'LA%')
              AND (agnr NOT LIKE '_BE%')
           THEN
               -- E => I : gleiche Nummer
               -- I => I : gleiche Nummer
               -- I => E : fortlaufende Nummer
               --IF astat IN ('E') OR (astat = 'I' AND ldcode = 'I') THEN
                  Nummer := agnr;           -- Bestellnummer = Auftragsnummer
               -- END IF;
           ELSE                          -- Beginnt mit AG...
               IF (ldcode = 'I') THEN
                   Nummer       := 'PA' || Substring(agnr FROM 3); -- Auftragsnummer beginnt mit 'AG'
                   --
                   IF (agnr LIKE 'FBE%') THEN
                      Nummer    := 'PA' || Substring(agnr FROM 4);  --F-ABK
                   END IF;
                   --
                   IF astat = 'E' AND agpos IS NOT NULL THEN --Projektfertigung
                      IF (SELECT ag_ownabk FROM auftg WHERE ag_astat = 'E' AND ag_nr = agnr AND ag_pos = agpos) IS NOT NULL THEN --der Externe Auftrag hat bereits eine ag_ownabk
                         Nummer := 'PP' || Substring(agnr FROM 3); -- Projektfertigung
                      END IF;
                   END IF;
               END IF;
               --
               IF (ldcode = 'E') THEN
                   Nummer       := 'BE' || Substring(agnr FROM 3); -- Auftragsnummer beginnt mit 'AG'
               END IF;
               --
           END IF;

           -- Setting aktiv: Auftragsposition an Bestellnummer anhängen, Auf Kundenauftrag bestellt und interne Bestellung gewünscht
           IF     TSystem.Settings__Getbool('ld_auftg=ag_nr_add_pos')
              AND (agpos IS NOT NULL)
              AND (Nummer IS NOT NULL)
              AND (astat IN ('E', 'R'))
              AND (ldcode = 'I')
           THEN
               Nummer := Nummer || '/' || LPAD(agpos::varchar,2,'0');
           END IF;
          --
       END IF;

       -- und raus wenn wir eine Nummer haben.
       IF Nummer IS NOT NULL THEN -- RETURN
          RAISE NOTICE 'EXIT';
          RETURN;
       END IF;

       -- Nummernkreis für R und I
       IF ldcode IN ('R', 'I') THEN
          -- Rahmenbestellnummer
          IF (ldcode = 'R') THEN
             NumCircle := 'ldsdokr';
          END IF;
          -- Lagerauftrag / Produktionsauftrag
          IF (ldcode = 'I') THEN
              NumCircle := 'ldsdoki';
          END IF;
          --
          SELECT * INTO rec FROM tsystem.getnumcirclenr_data(NumCircle, previewNummer);
          nummer := rec.result;
       END IF;

       -- Nummernkreis für E ODER keine separate Nummer für I
       IF    (  ldcode = 'E')
          OR ( (ldcode = 'I') AND (nummer IS NULL) )
       THEN
           SELECT * INTO rec FROM tsystem.getnumcirclenr_data('ldsdok', previewNummer);
           nummer    := rec.result;
           nummer    := nullIF( nummer, '' );
           NumCircle := 'ldsdok';
       END IF;

       -- Wir haben einen Auftragsbezug, dann ist es KEIN LA, sondern ein PA - bzw KEIN FBE sonder BE - mit fortlaufender Nummer
       -- auch nach mehrfachen Überlegungen: durch die Softwarephilosophie mit Sammelbedarfen ergibt ein direkter sprechender Bezug keinen Sinn mehr.
       -- Somit auch für die Kopfebene eine neue PA-Nummer pro Verkaufsposition und nicht die Sonderlösung mit /Pos.
       IF agnr IS NOT NULL THEN
          IF (ldcode = 'I') THEN
              nummer := REPLACE( rec.result,  'LA', 'PA' );
          ELSIF (ldcode = 'E') THEN
              nummer := REPLACE( rec.result, 'FBE', 'BE' );
          END IF;
       END IF;
       --
       RETURN;
    END $$ LANGUAGE plpgsql;
 --
 -- BestellNr zusammenfassen
 CREATE OR REPLACE FUNCTION twawi.ldsdok_GenConcatNr(r ldsdok) RETURNS varchar(75) AS $$
   SELECT r.ld_code || ' | ' || r.ld_auftg || ' | ' || r.ld_pos;
   $$ LANGUAGE SQL STABLE;

 CREATE OR REPLACE FUNCTION twawi.ldsdok_GenConcatNr(ldid integer) RETURNS varchar(75) AS $$
   SELECT twawi.ldsdok_GenConcatNr( ldsdok ) FROM ldsdok WHERE ld_id = ldid;
   $$ LANGUAGE SQL STABLE;
 --
--

-- In Teillieferungen zur Bestellposition erfasste Menge.
CREATE OR REPLACE FUNCTION TEinkauf.Sum_Teillieferung(
      -- Bestellpositions-ID
      ldid integer,
      -- Gelieferte Menge aufsummieren
      liefermenge boolean = false,
      -- Menge der automatischen Restposition mit aufsummieren
      includeRestpos boolean = false
  ) RETURNS numeric AS $$
  DECLARE
      s numeric;
  BEGIN
      IF liefermenge THEN

          SELECT sum( ldl_stkl )
            INTO s
            FROM ldslieferung
           WHERE ldl_ld_id = ldid
             AND CASE WHEN includeRestpos
                      THEN true
                      ELSE NOT ldl_restpos
                 END
          ;

      ELSE

          SELECT sum( ldl_stk )
            INTO s
            FROM ldslieferung
            WHERE ldl_ld_id = ldid
              AND CASE
                  WHEN includeRestpos
                      THEN true
                      ELSE NOT ldl_restpos
                  END
          ;

      END IF;

      RETURN coalesce(s,0);

 END $$ LANGUAGE plpgsql STABLE RETURNS NULL ON NULL INPUT;
--

CREATE OR REPLACE FUNCTION TEinkauf.epreisstaffel__anfartlist__generate__menge_agg_displaytext(
      id integer
  ) RETURNS varchar AS $$

      SELECT
          string_agg(
              est_mengevon ||
              coalesce(
                  '..' || est_mengebis,
                  ''
              ),
              '; '
              ORDER BY est_mengevon
          )
      FROM epreisstaffel
      WHERE est_aart_id = id
 $$ LANGUAGE sql;

CREATE OR REPLACE FUNCTION TEinkauf.epreisstaffel__epreis__generate__menge_agg_displaytext(
      id integer
  ) RETURNS varchar AS $$
      SELECT
          e_stk ||
          coalesce(
              ' ;' ||
              string_agg(
                  est_mengevon ||
                  coalesce('..' || est_mengebis, ''),
                  '; '
                  ORDER BY est_mengevon
              ),
              ''
          )
      FROM epreis
      LEFT JOIN epreisstaffel ON est_e_id = e_id
      WHERE e_id = id
      GROUP BY e_stk

 $$ LANGUAGE sql;


-- FUNCTIONS: beleg_x__anfrage__create__from .... BANF, AUFTG,
 CREATE OR REPLACE FUNCTION TEinkauf.Create_AnfrageFromBanf(
      anfnr varchar,
      bestanfposid integer
  ) RETURNS boolean AS $$
  DECLARE
      ids integer[];
  BEGIN

      ids[1] := bestanfposid;
      RETURN TEinkauf.Create_AnfrageFromBanf( anfnr, ids );
  END $$ LANGUAGE plpgsql VOLATILE;
 --

 CREATE OR REPLACE FUNCTION TEinkauf.Create_AnfrageFromBanf(
      anfnr varchar,
      bestanfposids integer[] ) RETURNS boolean AS $$
 DECLARE
      bapRec         record;
      newAnf         boolean;
      bapid          integer;
      i              integer;
      aartid         integer;
      maxAnfPos      integer;
      norm           varchar(40);
      agnr           varchar(40);
      projNr         varchar(50);
      apint          varchar(50);
   BEGIN

     -- TODO: Zusammenfassen von gleichen Banfpositionen. Das war in der Übernahme BANF=>Anfrage aus dem Anfragemodul heraus implementiert.
     -- Die DB Funktion ist wesentlich flexibler, kann aber nicht zusammenfassen. Bei Bedarf muss das hier nachgesetzt werden. (LG, 01/2014)
     IF
           bestanfposids IS NULL
        OR array_upper( bestanfposids, 1 ) = 0
     THEN
         RETURN false;
     END IF;

     -- Anfragekopf anlegen, falls noch keiner da ist. Ansonsten hängen wir an.
     INSERT INTO anfrage (anf_nr, anf_bisdatum, anf_termweek, anf_txt, anf_txt_rtf )
         SELECT anfnr, ba_dat, ba_termweek, ba_txt, ba_txt_rtf
             FROM bestanfpos JOIN bestanftxt ON bap_banr = ba_nr
             WHERE bap_id = bestanfposids[1]
               AND NOT EXISTS(SELECT true FROM anfrage WHERE anf_nr = anfnr )
     ;

     newAnf := FOUND;

     --Positionen mit Artikeln, Mengen, Preisen übernehmen
     maxAnfPos := max( aart_pos ) FROM anfart WHERE aart_anf_nr = anfNr;
     maxAnfPos := coalesce( maxAnfPos, 0 );

     --LOOP über Array, damit die Reihenfolge so erhalten bleibt, wie sie im Array angegeben wurde
     FOR i IN array_lower( bestanfposids, 1 ) .. array_upper( bestanfposids, 1 ) LOOP

         bapid     := bestanfposids[i];
         maxAnfpos := maxAnfpos+1;
         aartid    := null;

         SELECT bap_aknr, coalesce(bap_akbez, ak_bez) AS akbez, anfnr, bap_menge, coalesce(bap_konto, ak_awko, ac_konto) AS bap_konto, bap_ks, bap_mecod, bap_lkn,
                bap_txt, trim(bap_txt_rtf) AS bap_txt_rtf, bap_txtint, trim(bap_txtint_rtf) AS bap_txtint_rtf, bap_termin, bap_termweek, maxAnfpos,
                bap_minr, bap_agnr, bap_an_nr, ag_nident, bap_op_ix, bap_o2_n, bap_o2_id
             INTO baprec
             FROM bestanfpos
                 LEFT JOIN art ON ak_nr = bap_aknr
                 LEFT JOIN artcod ON ac_n = ak_ac
                 LEFT JOIN auftg ON ag_astat = 'E' AND ag_nr = bap_agnr AND ag_pos = bap_agpos
             WHERE bap_id = bapid
         ;

         -- Anfragepositionen einfügen aus Bestellanforderungspositionen
         INSERT INTO anfart (aArt_ak_nr, aArt_bez, aArt_anf_nr, aArt_menge, aArt_konto, aArt_ks, aArt_m_id,
             aArt_txt, aArt_txt_rtf, aArt_txtint, aArt_txtint_rtf, aArt_termin, aArt_termweek,aart_pos,
             aArt_op_ix, aArt_o2_n, aArt_o2_id)
         VALUES ( baprec.bap_aknr, baprec.akbez, baprec.anfnr, baprec.bap_menge, baprec.bap_konto, baprec.bap_ks, baprec.bap_mecod,
                  baprec.bap_txt, baprec.bap_txt_rtf, baprec.bap_txtint, baprec.bap_txtint_rtf, baprec.bap_termin, baprec.bap_termweek, maxAnfpos,
                  baprec.bap_op_ix,baprec.bap_o2_n,baprec.bap_o2_id)
         RETURNING aart_id INTO aartid;

         IF aartid IS NOT NULL THEN UPDATE bestanfpos SET bap_aart_id = aartid WHERE bap_id = bapid; END IF;

         -- Merken um ggf. Anfragekopfdaten noch nachzutragen ...
         IF norm   IS NULL THEN norm  :=baprec.ag_nident; END IF;
         IF agnr   IS NULL THEN agnr  :=baprec.bap_agnr;  END IF;
         IF apint  IS NULL THEN apint :=baprec.bap_minr; END IF;
         IF projNr IS NULL THEN projNr:=baprec.bap_an_nr;  END IF;

         INSERT INTO anflief(alief_anf_nr, alief_lkn)
             SELECT DISTINCT baprec.anfnr, baprec.bap_lkn
                 FROM bestanfpos
                 LEFT JOIN anflief al2 ON alief_anf_nr = baprec.anfnr AND alief_lkn = baprec.bap_lkn
             WHERE alief_id IS NULL
               AND (coalesce(baprec.bap_lkn,'')<>'')
               AND bap_id = bapid
         ;

         -- Nachtragen in Anfragekopfdaten ...
         UPDATE anfrage SET anf_agnr   = coalesce(anf_agnr,  agnr),
                            anf_nident = coalesce(anf_nident,norm),
                            anf_an_nr  = coalesce(anf_an_nr, projnr),
                            anf_apint  = coalesce(anf_apint,apint)
             WHERE anf_nr = baprec.anfnr AND ( (anf_agnr   IS DISTINCT FROM agnr)
                                          OR (anf_nident IS DISTINCT FROM norm)
                                          OR (anf_an_nr  IS DISTINCT FROM projnr)
                                          OR (anf_apint  IS DISTINCT FROM apint))
         ;

     END LOOP;

     RETURN newAnf;

   END $$ LANGUAGE plpgsql VOLATILE;
 --

 CREATE OR REPLACE FUNCTION TEinkauf.Create_AnfrageFromAuftg(
      anfnr varchar,
      agposid integer
  ) RETURNS boolean AS $$
  DECLARE
      ids integer[];
  BEGIN

      ids[1] := agposid;
      RETURN TEinkauf.Create_AnfrageFromAuftg( anfnr, ids );
  END $$ LANGUAGE plpgsql VOLATILE;
 --

 CREATE OR REPLACE FUNCTION TEinkauf.Create_AnfrageFromAuftg(
      anfnr varchar,
      agposids integer[]
  ) RETURNS boolean AS $$
  DECLARE
       agRec         record;
       newAnf         boolean;
       agid          integer;
       i              integer;
       aartid         integer;
       maxAnfPos      integer;
       norm           varchar(40);
       agnr           varchar(40);
       projNr         varchar(50);
       apint          varchar(50);
  BEGIN
     IF
          agposids IS NULL
       OR array_upper( agposids, 1 ) = 0
     THEN
         RETURN false;
     END IF;

     -- Anfragekopf anlegen, falls noch keiner da ist. Ansonsten hängen wir an.
     INSERT INTO anfrage (anf_nr, anf_agnr, anf_bisdatum, anf_termweek, anf_txt, anf_txt_rtf )
         SELECT anfnr, ag_nr, ag_kdatum, week_of_year(ag_kdatum), ag_txt, ag_txt_rtf
             FROM auftg WHERE ag_id = agposids[1]
               AND NOT EXISTS(SELECT true FROM anfrage WHERE anf_nr = anfnr )
     ;

     newAnf := FOUND;

     --Positionen mit Artikeln, Mengen, Preisen übernehmen
     maxAnfPos := max( aart_pos ) FROM anfart WHERE aart_anf_nr = anfNr;
     maxAnfPos := coalesce( maxAnfPos, 0 );

     --LOOP über Array, damit die Reihenfolge so erhalten bleibt, wie sie im Array angegeben wurde
     FOR i IN array_lower( agposids, 1 ) .. array_upper( agposids, 1 ) LOOP

         agid := agposids[i];
         maxAnfpos := maxAnfpos+1;
         aartid := null;

         SELECT
            ag_aknr,
            coalesce( ag_akbz, ak_bez ) AS akbez,
            anfnr,
            ag_stk_uf1,
            coalesce( ag_konto, ak_awko, ac_konto ) AS ag_konto,
            ag_ks,
            ( SELECT m_mgcode FROM artmgc WHERE m_id = ag_mcv ) AS me_id,
            ag_lkn,
            ag_postxt,
            trim( ag_postxt_rtf ) AS ag_postxt_rtf,
            ag_kdatum,
            week_of_year( ag_kdatum ) AS termweek,
            maxAnfpos,
            ag_nr,
            ag_an_nr,
            ag_nident,
            ag_kontakt
         INTO agrec
         FROM auftg
         LEFT JOIN art ON ak_nr = ag_aknr
         LEFT JOIN artcod ON ac_n = ak_ac
         WHERE ag_id = agid;

         -- Anfragepositionen einfügen aus Auftragspositionen
         INSERT INTO anfart (aArt_ak_nr, aArt_bez, aArt_anf_nr, aArt_menge, aArt_konto, aArt_ks, aArt_m_id,
             aArt_txt, aArt_txt_rtf, aArt_termin, aArt_termweek,aart_pos)
         VALUES ( agrec.ag_aknr, agrec.akbez, agrec.anfnr, agrec.ag_stk_uf1, agrec.ag_konto, agrec.ag_ks, agrec.me_id,
                  agrec.ag_postxt, agrec.ag_postxt_rtf, agrec.ag_kdatum, agrec.termweek, maxAnfpos)
         RETURNING aart_id INTO aartid;

         -- Merken um ggf. Anfragekopfdaten noch nachzutragen ...
         IF norm   IS NULL THEN norm   := agrec.ag_nident; END IF;
         IF agnr   IS NULL THEN agnr   := agrec.ag_nr;  END IF;
         IF apint  IS NULL THEN apint  := agrec.ag_kontakt; END IF;
         IF projNr IS NULL THEN projNr := agrec.ag_an_nr;  END IF;

         INSERT INTO anflief(alief_anf_nr, alief_lkn)
             SELECT DISTINCT agrec.anfnr, agrec.ag_lkn
                 FROM auftg
                 LEFT JOIN anflief al2 ON alief_anf_nr = agrec.anfnr AND alief_lkn = agrec.ag_lkn
             WHERE alief_id IS NULL
               AND coalesce( agrec.ag_lkn, '' ) <> ''
               AND ag_id = agid
         ;

         -- Nachtragen in Anfragekopfdaten ...
         UPDATE anfrage SET anf_agnr   = coalesce(anf_agnr,  agnr),
                            anf_nident = coalesce(anf_nident,norm),
                            anf_an_nr  = coalesce(anf_an_nr, projnr),
                            anf_apint  = coalesce(anf_apint,apint)
             WHERE anf_nr = agrec.anfnr AND ( (anf_agnr   IS DISTINCT FROM agnr)
                                          OR (anf_nident IS DISTINCT FROM norm)
                                          OR (anf_an_nr  IS DISTINCT FROM projnr)
                                          OR (anf_apint  IS DISTINCT FROM apint))
         ;

     END LOOP;

     RETURN newAnf;

   END $$ LANGUAGE plpgsql VOLATILE;


-- FUNCTIONS: beleg_x__einkauf__create__from .... BANF
 -- Erzeugt eine Bestellung mit der angegebenen Nummer für die BANF-Position mit der angegebenen ID
 CREATE OR REPLACE FUNCTION TEinkauf.Create_BestellungFromBanf(
      ldcode varchar(1),
      ldauftg varchar,
      bestanfposid integer
  ) RETURNS boolean AS $$
  DECLARE
      ids integer[];
  BEGIN

     ids[1] := bestanfposid;
     RETURN TEinkauf.Create_BestellungFromBanf( ldauftg, ids );
  END $$ LANGUAGE plpgsql VOLATILE;
 --

 -- Erzeugt eine gemeinsame Bestellung mit der angegebenen Nummer für alle BANF-Positionen, deren ID in bestanfposIDs enthalten ist.
 CREATE OR REPLACE FUNCTION TEinkauf.Create_BestellungFromBanf(
      ldcode varchar(1),
      ldauftg varchar,
      bestanfposids integer[]
  ) RETURNS boolean AS $$
  DECLARE
      sr                record;
      lddbrid           varchar;
      ldid              varchar;
      nCheck            record;
      newBest           boolean;
      bapid             integer;
      preis             TWawi.Preis;
      rhl_nr            varchar;
      rhl_stko          numeric;
      ldterm            date;
      ldtermweek        varchar;
      ldkontakt         varchar;
      ldbem             varchar;
      epreisdbrid       varchar;
      epreisid          varchar;
      ldPos             integer;
      i                 integer;
      RMengeAusreichend boolean;
      error_info        varchar;
  BEGIN
     -- Kontrollmerkmale Artikel?

     IF
            bestanfposids IS NULL
         OR array_upper( bestanfposids, 1 ) = 0
     THEN
        RETURN false;
     END IF;

    -- Prüft ob Mengeneinheit der Banf-Position und des zugewiesenen Preises valide sind für den Artikel. Beispielausgabe:
    -- Ungültige Mengeneinheit für Artikel NEUERARTIKEL in TEST Pos. 2. GME Artikel=Stück2, Banf-ME=Stück, Preis-ME=?
    -- Ungültige Mengeneinheit für Artikel MAT-TEST in TESTANFRAGE Pos. 3. GME Artikel=Meter, Banf-ME=Stück, Preis-ME=Packung

     SELECT trim(string_agg(fail, E'\r\n')) INTO error_info
     FROM ( SELECT DISTINCT format( lang_text(13854), bap_aknr, bap_banr, bap_pos, standard_mgc(bap_aknr), lang_artmgc(bap_mecod), coalesce( lang_artmgc_id(me2.m_id),'?')) AS fail
              FROM bestanfpos
              LEFT JOIN preise ON bestanfpos.dbrid = target_dbrid AND target_table = 'bestanfpos'
              LEFT JOIN artmgc AS me1 ON me1.m_mgcode = bap_mecod AND m_ak_nr = bap_aknr
              LEFT JOIN artmgc AS me2 ON preise IS NOT NULL AND ( me2.m_id = (preise.preis).artmgcid)
              WHERE bap_id = ANY(bestanfposids) AND ( ( preise.preis IS NOT NULL AND me2.m_id IS NULL) OR (me1.m_id IS NULL))
              ) as sq1;

     IF coalesce( error_info, '' ) <> '' THEN
       RAISE EXCEPTION '%', lang_text(16455) || E'\n\n' || error_info;
     END IF;

     newBest := NOT EXISTS(SELECT true FROM ldsdok WHERE ld_auftg = ldauftg AND ld_code = ldcode);

     --LOOP über Array, damit die Reihenfolge so erhalten bleibt, wie sie im Array angegeben wurde
     FOR i IN array_lower( bestanfposids, 1 ) .. array_upper( bestanfposids, 1 ) LOOP

       bapid       := bestanfposids[ i ];
       ldPos       := TWaWi.ldsdok__ld_pos__next( ldcode, ldauftg );
       epreisdbrid := null;
       rhl_nr      := null;
       ldterm      := null;
       ldtermweek  := null;
       ldbem       := null;
       lddbrid     := null;

       --Ausgangsdaten für Position sammeln
       SELECT  bestanfpos.*, bestanftxt.ba_txt, artmgc.*, auftg.ag_id, auftg.ag_nident, auftg.ag_aknr, art.ak_norm, target_table,
               -- #7715
               (tartikel.adtx_getArtTxtLang(ak_nr, 'EK', bap_lkn)).txt    AS ak_bestxt,
               (tartikel.adtx_getArtTxtLang(ak_nr, 'EK', bap_lkn)).txtrtf AS ak_bestxt_rtf,
               target_dbrid, preise.preis, adk2.*,
               artinfo.ain_lagzu_Txt, artinfo.ain_lagzu_txt_rtf, art.ak_los
       INTO sr
       FROM bestanfpos
         JOIN bestanftxt   ON ba_nr = bap_banr
         LEFT JOIN art     ON ak_nr = bap_aknr
         LEFT JOIN artinfo ON ak_nr = ain_ak_nr
         LEFT JOIN artcod  ON ac_n = ak_ac
         LEFT JOIN artmgc  ON bap_mecod = m_mgcode AND m_ak_nr = bap_aknr
         LEFT JOIN auftg   ON ag_astat = 'E' AND ag_nr = bap_agnr AND ag_pos = bap_agpos
         LEFT JOIN adk2    ON bap_lkn = a2_krz
         LEFT JOIN preise  ON bestanfpos.dbrid = target_dbrid AND target_table = 'bestanfpos' -- Preis-datensatz, falls nicht manuell festgelegt
       WHERE bap_id = bapid;

       -- Bestellung kann nicht angelegt werden. Für BANF % Pos. % wurde bereits eine Bestellung erzeugt.
       IF sr.bap_ld_id IS NOT null THEN
         RAISE EXCEPTION '%', TSystem.formatS(lang_text(13596), sr.bap_banr, sr.bap_pos::varchar);
       END IF;

       preis := sr.Preis;

       -- Das wird ein Produktionsauftrag. Da muss kein Lieferant angegeben sein. Wir gehen einfach von Raute aus und setzen den nach Anlage der Bestellung auch in bap_lkn.
       IF (ldcode = 'I') AND ( coalesce(sr.bap_lkn,'') = '') THEN
         sr.bap_lkn := '#';
       END IF;

       --Preis/Lieferant/Menge war manuell festgelegt, es gibt keinen "richtigen" Preiseintrag
       IF (preis IS NULL) OR (NOT preis.IsValid) THEN
         -- Leeren Preis mit Einkaufsvorgaben erzeugen / Steuercodes, Ak_los als Losgröße, StandardMGC-Id etc...
         preis := TWawi.Create_Preis(srcTable => 'art',
                                     srcDBRID => art.dbrid,
                                     mengeGME => sr.bap_menge) FROM art WHERE ak_nr = sr.bap_aknr; -- #13024 bei manueller Preisanlage muss zwingend die Menge zur Anlage mitgegeben werden, sonst Fehler Ermittlung Losmenge
         --
         preis.preis                  := sr.bap_ep_uf1_basisw;             -- Preis aus Bestanfpos ist immer in Basis-Währung, kann aber abweichende ME haben
         preis.preis_uf1              := preis.preis * sr.m_uf;            -- Umrechnung nach UF1
         preis.preis_uf1_basisw       := preis.preis_uf1;                  -- Bleibt, weil Basis_W
         preis.preis_uf1_basisw_abzu  := sr.bap_ep_uf1_basisw;             -- Bleibt, weil keine Abzu

         preis.artmgcid               := sr.m_id;
         preis.ad_krz                 := sr.bap_lkn;
         preis.rabatt                 := sr.bap_rab;
         preis.IsValid                := True;
       END IF;

       -- Lieferdatum (bei angegebener Lieferfrist) und Bemerkung aus Epreis holen
       IF preis.source_dbrid IS NOT NULL THEN

         -- Liefertermin, wenn heute bestellt wird.
         ldterm := TWawi.GetLieferTermin(preis.source_table, preis.source_dbrid, current_date);
         ldtermweek := termweek(ldterm);

         CASE
           WHEN ((preis.source_table = 'ldsdok') AND (ldcode='E')) THEN --Wird normale Bestellung, Kam aus Rahmenbestellung
             SELECT (ld_auftg||'/'||ld_pos), (rahmen_stk_ldsdok_offen(ld_auftg || '/' || ld_pos)).stko INTO rhl_nr, rhl_stko -- Offene Menge
             FROM  ldsdok JOIN art ON ld_aknr = ak_nr
             WHERE ldsdok.dbrid = preis.source_dbrid;

           WHEN (preis.source_table = 'epreis') OR  (preis.source_table = 'epreisstaffel') THEN
             SELECT epreis.dbrid, epreis.e_id INTO epreisdbrid, epreisid
             FROM epreis JOIN epreisstaffel ON est_e_id = e_id
             WHERE coalesce(epreisstaffel.dbrid, epreis.dbrid) = preis.source_dbrid
             LIMIT 1;
           ELSE
         END CASE;
       END IF;

       -- [SF AO 1] Bei AutoOrder wird ak_los bevorzugt verwendet, wenn Lieferantenpreis. Bei Rahmen wird ld_eklos vom Rahmen genutzt.
       IF sr.bap_autoorder THEN
         IF ( (preis.source_table = 'epreis') OR  (preis.source_table = 'epreisstaffel')) THEN
           preis.los := tartikel.me__menge_uf1__in__menge(preis.artmgcid, sr.ak_los); -- tartikel.me__menge_uf1__in__menge(mce, stk)  --An dieser Stelle umrechnen, damit GME in BestellME umgerechnet wird
         END IF;
         ldkontakt:=null; -- Zuständiger Mitarbeiter der Bestellung soll leer bleiben. Muss explizit null gesetzt werden, da sonst ldsdok = current_user greift
       ELSE
         ldKontakt:=current_user;
       END IF;

       -- Ohne Bestelltext der Banf-Position, den Bestelltext aus Artikelstamm nehmen.
       IF nullIf(sr.bap_txt,'') IS NULL AND nullIf(sr.bap_txt_rtf,'') IS NULL THEN
         sr.bap_txt     := sr.ak_bestxt;
         sr.bap_txt_rtf := sr.ak_bestxt_rtf;
       END IF;

       -- Wenn kein Einlagerungshinweis in BANF-Pos angg. ist, dann Fallback auf Artikelstamm-Einträge.
       IF nullIf(sr.bap_lagzu_txt,'') IS NULL AND nullIf(sr.bap_lagzu_txt_rtf,'') IS NULL THEN
         sr.bap_lagzu_txt      := sr.ain_lagzu_txt;
         sr.bap_lagzu_txt_rtf  := sr.ain_lagzu_txt_rtf;
       END IF;

       -- Lieferdatum nur vorschlagen, wenn in Banf-Pos. gar nichts angegeben ist.
       IF sr.bap_termweek IS NULL AND sr.bap_termin IS NULL THEN
         sr.bap_termin   := ldterm;
         sr.bap_termweek := ldtermweek;
       END IF;

       --Bestellposition anlegen
       INSERT INTO ldsdok (
         ld_code, ld_auftg, ld_pos, ld_rhl_nr, ld_kn, ld_aknr, ld_akbz,
         ld_ekref,
         ld_stk, ld_mce, ld_eklos,
         ld_kontakt, ld_konto, ld_ks,
         ld_term, ld_termweek, ld_arab, ld_an_nr,ld_nident,
         ld_ep, ld_ep_uf1,
         ld_waer, ld_kurs, ld_steucode, ld_steuproz,
         ld_ag_id, ld_bem, ld_txt, ld_txt_rtf, ld_txtint, ld_txtint_rtf,
         ld_lagzu_txt, ld_lagzu_txt_rtf)
       VALUES( ldcode, ldauftg, ldpos,rhl_nr, sr.bap_lkn, sr.bap_aknr,sr.bap_akbez,
         ifthen(TSystem.Settings__Getbool('BANF_AsEKRef'), sr.bap_banr, null), -- Einkaufsreferenz soll zusätzlich die Banf-Nummer führen. LG meint das ist Quatsch weil das ja in ld_banfnr schon steht, baut es aber ein.
                                                                               -- siehe http://redmine.prodat-sql.de/issues/6946 (Workorder) LG => Bemängelt von anderen Kunden, schaltbar gemacht.
         coalesce(preis.Menge_los, preis.menge), preis.artmgcid, preis.los, -- 12931: menge_los statt menge. in Menge steht die komplette Rahmenbestellungsmenge drin
         ldkontakt, sr.bap_konto, sr.bap_ks,
         sr.bap_termin, sr.bap_termweek, coalesce(sr.bap_rab,0), sr.bap_an_nr, coalesce(sr.ag_nident,sr.ak_norm),
         preis.preis, preis.preis_uf1,
         preis.wacode, preis.kurs, preis.steucode, preis.steuproz,
         ifthen(sr.ag_aknr = sr.bap_aknr, sr.ag_id, null), -- Nur mit Auftrag verknüpfen, wenn Artikel identisch ist.
         preis.aknr_referenz, sr.bap_txt, sr.bap_txt_rtf, sr.bap_txtint, sr.bap_txtint_rtf,
         sr.bap_lagzu_txt, sr.bap_lagzu_txt_rtf)
       RETURNING dbrid, ld_id INTO lddbrid, ldid;

       IF lddbrid IS NOT NULL THEN --Bestell-ID und Datum ggf. in Bestellanforderungsposition zurückschreiben
         UPDATE bestanfpos SET bap_ld_id    = ld_id,
                               bap_lkn      = coalesce(bap_lkn, ifthen(ldcode='I','#',bap_lkn)), -- Bei Anlage Fertigungsauftrag, Internkürzel '#' in Banfpos zurückschreiben.
                               bap_termin   = coalesce(bap_termin, sr.bap_termin),
                               bap_termweek = coalesce(bap_termweek, sr.bap_termweek),
                               bap_done     = ifthen(bap_autoorder, true, bap_done)              -- Auto-Order Banf wird nach dem Bestellen geschlossen, sonst lassen wir es so, wie es ist.
         FROM ldsdok WHERE ldsdok.dbrid = lddbrid AND bap_id = bapid;
         -- legt auch ldsauftg an?!

         UPDATE ldsdoktxt SET ldt_txt = sr.ba_txt WHERE ldt_code = ldcode AND ldt_auftg = ldauftg AND ldt_txt IS NULL; -- Wurde vom ldsdok - Trigger angelegt. Beststellanforderungs-Kopftext weiterschleifen

         IF preis.source_dbrid IS NOT NULL THEN
           --Ab/Zuschläge kopieren aus Rahmen oder Lieferantendaten, falls welche hinterlegt sind.
           IF preis.hasAbzu THEN
             CASE
               WHEN ((preis.source_table = 'ldsdok') AND (ldcode='E')) THEN
                 --PERFORM TWawi.Create_Abzu('ldsdok', preis.source_dbrid, 'ldsdok', lddbrid);
                 PERFORM TWawi.Abzu_Copy('Ldsdok_Abzu', (SELECT ld_id FROM ldsdok WHERE dbrid = preis.source_dbrid)::varchar, 'Ldsdok_Abzu', ldid, preis.Menge);

               WHEN preis.source_table = 'epreis' THEN
                 --PERFORM TWawi.Create_Abzu('epreis', preis.source_dbrid, 'ldsdok', lddbrid);
                 PERFORM TWawi.Abzu_Copy('Epreis_Abzu', (SELECT e_id FROM epreis WHERE dbrid = preis.source_dbrid)::varchar, 'Ldsdok_Abzu', ldid, preis.Menge);

               WHEN preis.source_table = 'epreisstaffel' THEN
                 --PERFORM TWawi.Create_Abzu('epreis', epreisdbrid, 'ldsdok', lddbrid);
                 PERFORM TWawi.Abzu_Copy('Epreis_Abzu', epreisid, 'Ldsdok_Abzu', ldid, preis.Menge);

               ELSE
             END CASE;
           END IF;
         END IF;
       END IF;

     END LOOP;

     RETURN newBest;
  END $$ LANGUAGE plpgsql VOLATILE;
 --
--


--
CREATE OR REPLACE FUNCTION TEinkauf.Set_Mahnstufe(
      dokunr integer,
      mahnstufe integer
  ) RETURNS void AS $$
  BEGIN

      IF mahnstufe = 1 THEN

        UPDATE ldsdokdokutxt SET
            ltd_mahndat = current_date,
            ltd_mahnstufe = mahnstufe,
            ltd_mahntxt = belarzu__zu_tit__gettxt('DOKEINK_TXT_KOPF_MAHN1', 'D', false),
            ltd_mahntxt_rtf = belarzu__zu_tit__gettxt('DOKEINK_TXT_KOPF_MAHN1', 'D', false),
            ltd_mahntxt1 = belarzu__zu_tit__gettxt('DOKEINK_TXT_FUSS_MAHN1', 'D', false),
            ltd_mahntxt1_rtf = belarzu__zu_tit__gettxt('DOKEINK_TXT_FUSS_MAHN1', 'D', false)
        WHERE ltd_dokunr = dokunr;

      END IF;
      --
      IF mahnstufe = 2 THEN

          UPDATE ldsdokdokutxt SET
            ltd_mahndat = current_date,
            ltd_mahnstufe = mahnstufe,
            ltd_mahntxt = belarzu__zu_tit__gettxt('DOKEINK_TXT_KOPF_MAHN2', 'D', false),
            ltd_mahntxt_rtf = belarzu__zu_tit__gettxt('DOKEINK_TXT_KOPF_MAHN2', 'D', false),
            ltd_mahntxt1 = belarzu__zu_tit__gettxt('DOKEINK_TXT_FUSS_MAHN2', 'D', false),
            ltd_mahntxt1_rtf = belarzu__zu_tit__gettxt('DOKEINK_TXT_FUSS_MAHN2', 'D', false)
          WHERE ltd_dokunr = dokunr;
      END IF;

  END $$ LANGUAGE plpgsql VOLATILE;
--


-- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!   Bestellvorschlag:  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
 -- FUNCTIONs bestellvorschlag__generate__from__sources (getBestVorschlag_Union) | Bestvorschlag aus Bedarfen generieren

  -- Alle zusammen
  -- DROP FUNCTION IF EXISTS TEinkauf.bestellvorschlag__generate__from__sources(varchar, varchar, date, varchar, varchar);
  CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlag__generate__from__sources(
        _flags_enum                 varchar,
        inAkac                      varchar,
        inBdate                     date,
        _ag_nr                      varchar,
        inLgort                     varchar
    )
    RETURNS TABLE(
        typ                         varchar,
        typindex                    integer,
        b_id                        integer,
        ak_nr                       varchar,
        ak_bez                      varchar,
        ak_tot                      numeric(12,4),
        ak_verfueg                  numeric(12,4),
        src_mgcode                  integer,
        standard_mgc_id             integer,
        b_date                      date,
        b_bestdat                   date,
        menge_uf1                   numeric,
        menge_los_uf1               numeric,
        bedarf_verursacher_list     tartikel.bedarf_verursacher[]
    )
    AS $$
    BEGIN

        _flags_enum := nullIf( trim( _flags_enum ), '' );
        _flags_enum := nullIf( _flags_enum, '%' );

        _ag_nr      := IFTHEN(_ag_nr = '', '%', _ag_nr); -- null?

        -- _flags_enum IS NULL => alle
        RETURN QUERY

                SELECT *
                FROM TEinkauf.bestellvorschlag__generate__from__Bedarf(inAkac, inBdate, _ag_nr)
                WHERE
                      _flags_enum IS NULL
                   OR TSystem.ENUM_GetValue( _flags_enum, 'doBedarfe' )
            UNION

                SELECT *
                FROM TEinkauf.bestellvorschlag__generate__from__melde__lag( inAkac, inLgort )
                WHERE
                      _flags_enum IS NULL
                  OR TSystem.ENUM_GetValue( _flags_enum, 'doMelde' )
            UNION
                SELECT *
                FROM TEinkauf.bestellvorschlag__generate__from__melde__art( inAkac )
                WHERE
                     -- ACHTUNG - sinnloser Prozentvergleich, da die Oberfläche sinnlos ein Prozent zurückgibt, wenn nichts eingegeben wurde. evtl mal anpassen
                     -- Meldeartikel nur anzeigen, wenn kein Lagerort, damit kein Durcheinander zB bei Werkzeugschrank entsteht.
                     TSystem.ENUM_GetValue( _flags_enum, 'doMelde' )
                 AND coalesce( inLgort, '%' ) = '%'

            UNION
                SELECT *
                FROM TEinkauf.bestellvorschlag__generate__from__verkauf( inAkac, _ag_nr )
                WHERE TSystem.ENUM_GetValue( _flags_enum, 'doAuftg' )

            UNION
                SELECT *
                FROM TEinkauf.bestellvorschlag__generate__from__banf( _flags_enum, inAkac, inBdate )
                WHERE
                     _flags_enum IS NULL
                     --alle oder gezielt Anfragen
                  OR TSystem.ENUM_GetValue( _flags_enum, 'doBANFAnfragen' )
                  OR TSystem.ENUM_GetValue( _flags_enum, 'doBANF' )
                     -- LOLL GIBTS DAS ÜBERHAUPT NOCH?!
                  OR TSystem.ENUM_GetValue( _flags_enum, 'doBANFoAuftg' );

    END $$ LANGUAGE plpgsql;
  --

  -- Typ 52+55+56: BANF (Anfragen)(Rahmen)
  -- DROP FUNCTION IF EXISTS TEinkauf.bestellvorschlag__generate__from__banf(varchar, varchar, date);
  CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlag__generate__from__banf(
        _flags_enum   varchar, -- = doBANFAnfragen siehe Funktionseinstieg
        inAkac        varchar,
        inBdate       date
    )
    RETURNS TABLE (
        typ                         varchar,
        typindex                    integer,
        b_id                        integer,
        ak_nr                       varchar,
        ak_bez                      varchar,
        ak_tot                      numeric(12,4),
        ak_verfueg                  numeric(12,4),
        src_mgcode                  integer,
        standard_mgc_id             integer,
        b_date                      date,
        b_bestdat                   date,
        menge_uf1                   numeric,
        menge_los_uf1               numeric,
        bedarf_verursacher_list     tartikel.bedarf_verursacher[]
    )
    AS $$
    BEGIN

       -- Bestellanforderungen: Anfragen

       -- Todo: eingangsflags validieren

       -- !!!!!!   Wenn über alle Bedarfe, dann nur Anfragen da in Bedarfen bereits BANF enthalten  !!!!!!
       IF    _flags_enum IS NULL
          OR TSystem.ENUM_GetValue(_flags_enum, 'doBedarfe')
       THEN
           _flags_enum := TSystem.ENUM_SetValue(_flags_enum, 'doBANFAnfragen');
       END IF;

       inAkac := coalesce(nullif(trim(inAkac), ''), '%');

       RETURN QUERY
       SELECT CASE WHEN    bap.bap_anfrage
                        OR bap.auswaerts
                   THEN
                   -- Anfragen
                   CASE WHEN bap_rahmen THEN
                          '55:Anfragen (BANF, Rahmen)'
                        WHEN bap.auswaerts THEN
                          '57:Anfragen (BANF, Auswärts)'
                        ELSE
                          '56:Anfragen (BANF)'
                    END
              ELSE
                   -- Richtige BANF (Mit / Ohne Auftragsbezug)
                   CASE WHEN bap.BanfOAuftg THEN -- Interne Auswertung weiter unten im SELECT
                          50 + ifthen(bap_rahmen, 1, 0) || ':BANF (OAuftg' || ifthen(bap_rahmen, ', Rahmen)', ')')
                        ELSE
                          52 + ifthen(bap_rahmen, 1, 0) || ':BANF' || ifthen(bap_rahmen, ' (Rahmen)', '')
                   END
              END::varchar
              AS typ, -- 55,56,50
              --
              CASE WHEN    bap.bap_anfrage
                        OR bap.auswaerts
                   THEN
                   CASE WHEN bap_rahmen THEN
                           55  -- '55:Anfragen (BANF, Rahmen)'
                        WHEN bap.auswaerts THEN
                           57  -- '57:Anfragen (BANF, Auswärts)'
                        ELSE
                           56  -- '56:Anfragen (BANF)'
                   END
              ELSE
                   CASE WHEN BanfOAuftg THEN
                           50 + ifthen(bap_rahmen, 1, 0) -- '50:BANF OAuftg' + 1 wenn Rahmen
                        ELSE
                           52 + ifthen(bap_rahmen, 1, 0) -- '52:BANF' + 1 wenn Rahmen
                   END
              END::integer
              AS typindex, --
              --
              null::integer,
              bap.aknr,
              coalesce(bap.bap_akbez, art.ak_bez),
              art.ak_tot,
              art.ak_verfueg,
              bap.bap_mecod,
              tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(art.ak_nr),
              bap.dat,
              timediff_substdays(bap.dat, ak_bfr, true)::date,
              bap.qty AS menge_uf1,
              -TArtikel.art__ak_los__qty_round(bap.aknr, -bap.qty) AS menge_los_uf1,
              bedarf_verursacher -- Liste der bedarfserzeugenden Aufträge inkl. zugehöriger Bedarfsmenge
         FROM (
              -- Summierung pro Artikel
              SELECT bap_anfrage,
                     bap_rahmen,
                     aknr,
                     bap_akbez,
                     bap_mecod,
                     array_remove(
                        array_agg(
                            CASE WHEN qty < 0
                                 THEN bedarf_verursacher
                            END
                            ORDER BY bapid
                        ),
                        null
                     )                                         AS bedarf_verursacher, -- ohne null, 3 Infos per eigenem Typ: Verursacher-Typ (Auftrag, BANF), ID und Bedarfsmenge
                     min(dat)                                  AS dat,
                     sum(qty)                                  AS qty,
                     string_agg(DISTINCT bap_ks, ';')          AS bap_ks,
                     string_agg(DISTINCT bap_konto, ';')       AS bap_konto,
                     bool_and(BanfOAuftg)                      AS BanfOAuftg,
                     auswaertsident IS NOT NULL                AS auswaerts
                FROM (-- Bestellanforderungen zu Aufträgen
                      SELECT bap_id                           AS bapid,
                             bap_aknr                         AS aknr,
                             bap_akbez,
                             bap_mecod,
                             coalesce( bap_termin,
                                       termweek_to_date(bap_termweek),
                                       current_date
                             )                                AS dat,
                             -( bap_menge_uf1 )               AS qty,
                             ( 2, bap_id, bap_menge_uf1 )::tartikel.bedarf_verursacher AS bedarf_verursacher,
                             bap_anfrage,
                             bap_rahmen,
                             bap_ks,
                             bap_konto,
                             bap_agnr IS NULL                 AS BanfOAuftg,
                             -- Banf Anfrage für Auswärtsbearbeitung. Keine Gruppierung / Aggregierung
                             bap_aknr || ' -> ' || NullIf( concat_ws( ':', bap_op_ix, bap_o2_n ), '' )  AS auswaertsident
                        FROM bestanfpos
                       WHERE true
                         AND NOT bap_done
                         AND bap_doeinkauf
                         AND ifthen(TSystem.ENUM_GetValue(_flags_enum, 'doBANFAnfragen'), bap_anfrage, True) --Nur Anfragen sind gewünscht.
                         AND NOT bap_prognose
                         AND bap_ld_id IS NULL -- Keine Bestellung vorhanden, die die BANF deckt
                         AND coalesce( bap_termin, termweek_to_date(bap_termweek), current_date ) <= coalesce( inBdate, current_date )
                       ORDER BY bap_aknr
                      ) AS src
               GROUP BY bap_anfrage, bap_rahmen, aknr, bap_akbez, bap_mecod, bap_ks, bap_konto, BanfOAuftg, auswaertsident  -- Summe pro Artikel
              ) AS bap
         LEFT JOIN art ON art.ak_nr = bap.aknr
        WHERE coalesce(art.ak_ac, '') LIKE inAkac
          AND bap.qty < 0
       ;
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- Typ 4: Auftragsbilanz
  -- #7448
  -- TODO auftgmatinfo agmi_beistell
  -- DROP FUNCTION IF EXISTS TEinkauf.bestellvorschlag__generate__from__verkauf(varchar, varchar);
  CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlag__generate__from__verkauf(
         IN inAkac varchar,
         IN inAuftg varchar
    )
    RETURNS TABLE(
        typ                         varchar,
        typindex                    integer,
        b_id                        integer,
        ak_nr                       varchar,
        ak_bez                      varchar,
        ak_tot                      numeric(12,4),
        ak_verfueg                  numeric(12,4),
        src_mgcode                  integer,
        standard_mgc_id             integer,
        b_date                      date,
        b_bestdat                   date,
        menge_uf1                   numeric,
        menge_los_uf1               numeric,
        bedarf_verursacher_list     tartikel.bedarf_verursacher[]
    )
    AS $$
    DECLARE auftragsnummern varchar[];
    BEGIN
      IF coalesce(inAuftg, '') = '' THEN
          RETURN;
      END IF;

      inAkac := coalesce(nullIf(trim(inAkac), ''), '%');

      -- Auftragsnummern und IDs aus Kundenaufträge anhand LIKE sammeln
      -- inkl. zugehörige Fertigungen mit entspr. Auftragsnummern der Materialbedarfe
      SELECT array_agg( DISTINCT auftg.ag_nr ) || array_agg( DISTINCT mat.ag_nr )
        INTO auftragsnummern
        FROM auftg
        LEFT JOIN ldsauftg ON la_ag_id = ag_id
        LEFT JOIN ldsdok ON     (   ld_ag_id = ag_id
                                 OR ld_id = la_ld_id
                                 )
                            AND ld_code = 'I'
        LEFT JOIN LATERAL tplanterm.get_all_child_abk(ld_abk) ON ld_abk IS NOT NULL
        LEFT JOIN auftg AS mat ON mat.ag_parentabk = get_all_child_abk
       WHERE auftg.ag_astat = 'E'
         AND auftg.ag_nr LIKE inAuftg
         AND NOT auftg.ag_done
         AND coalesce(NOT ldsdok.ld_done, true)
         AND coalesce(NOT mat.ag_done, true)
        -- AND NOT EXISTS(SELECT true FROM auftgmatinfo WHERE agmi_ag_id = auftg.ag_id AND agmi_beistell) -- ? (TODO auftgmatinfo agmi_beistell) muss doch trotzdem eingekauft werden?
      ;
      --
      IF auftragsnummern IS NULL THEN -- nichts gefunden, dann
          -- Auftragsnummern und IDs der Materialbedarfe aus Fertigungsauftrag anhand LIKE sammeln
          SELECT array_agg(DISTINCT mat.ag_nr)
            INTO auftragsnummern
            FROM ldsdok
            JOIN LATERAL tplanterm.get_all_child_abk(ld_abk) ON ld_abk IS NOT NULL
            JOIN auftg AS mat ON mat.ag_parentabk = get_all_child_abk
           WHERE ld_code = 'I'
             AND ld_auftg LIKE inAuftg
             AND NOT ld_done
             AND NOT mat.ag_done
          ;
      END IF;

      -- Auftragsbilanz über ermittelte Auftragsnummern
      RETURN QUERY
      SELECT '40:Auftrag'::varchar AS typ,
              40                   AS typindex,
             null::integer,
             art.ak_nr,
             -- Hinweis: Überschriebene Artikelbezeichnung aus Auftrag?! - nicht möglich: es müßte in den Bedarfen bereits gruppiert werden nach Artikelnummer und Bezeichnung
             -- Gruppierung wie in BANF anch ArtikelNr/Beizeichnung entfällt
             art.ak_bez,
             art.ak_tot,
             art.ak_verfueg,
             null::integer,
             tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(art.ak_nr),
             auftragbilanz.dat,
             timediff_substdays(auftragbilanz.dat, ak_bfr, true)::date,
             auftragbilanz.qty AS menge_uf1,
             -TArtikel.art__ak_los__qty_round(auftragbilanz.aknr, -auftragbilanz.qty) AS menge_los_uf1,
             bedarf_verursacher -- Liste der bedarfserzeugenden Aufträge inkl. zugehöriger Bedarfsmenge
       FROM (-- Summierung pro Artikel, Ausschluss gedeckter ag_id in Liste
             SELECT aknr,
                    array_remove(
                       array_agg(
                           CASE WHEN qty < 0
                                THEN bedarf_verursacher
                           END
                           ORDER BY agid, bapid
                       ),
                       null
                    ) AS bedarf_verursacher, -- ohne null, 3 Infos per eigenem Typ: Verursacher-Typ (Auftrag, BANF), ID und Bedarfsmenge
                    min(dat) AS dat, sum(qty) AS qty
               FROM (
                     -- Bestellanforderungen zu Aufträgen
                     SELECT null::integer AS agid, bap_id AS bapid, bap_aknr AS aknr, coalesce(bap_termin, current_date) AS dat, -(bap_menge_uf1) AS qty, (2, bap_id, bap_menge_uf1)::tartikel.bedarf_verursacher AS bedarf_verursacher
                       FROM bestanfpos
                      WHERE bap_agnr = ANY (auftragsnummern)
                        AND NOT bap_done
                        AND NOT bap_anfrage
                        AND NOT bap_prognose
                        AND bap_ld_id IS NULL -- Keine Bestellung vorhanden, die die BAN13F deckt

                     UNION
                     -- eigtl. Auftragsbilanz (offene Aufträge und deren Deckung durch direkt und indirekt verknüpfte Bestellungen)
                         -- Achtung m:n-Beziehung!
                         -- Die Schwierigkeit ist: Welche Aufträge sind mit welchen Bestellungen erledigt bzw. mit welchen Mengen gedeckt und dürfen nicht in der AG-ID-Liste (Array ag_ids) sein. Denn diese Liste wird in Bestellvorschlägen übergeben und die folg. Bestellung mit den entspr. Aufträgen verknüpft.
                         -- Wenn Deckung an der Auftragsposition erreicht ist, wird die ID nicht ausgegeben.
                     SELECT agid, null, aknr, dat, -qty, (1, agid, qty)::tartikel.bedarf_verursacher
                       FROM tauftg.auftg_ldsdok_bilanz(VARIADIC auftragsnummern)
                      WHERE qty > 0 -- Überlieferungen (größere Deckung durch Bestellung) eines Auftrags können nicht sinnvoll anhand Artikel aufsummiert werden.
                                   -- Auftragsbedarf würde durch unzugeordnete Bestellung (auf Basis der Summe über Artikel) (teil-)gedeckt werden.
                                   -- Bsp: Auftg A mit Art 1 ist mit +10 "überliefert", Auftg B mit Art 1 hat Bedarf von 15 (ohne zugeordnete Bestellung). Würde über Artikelsumme Bedarf von 5 ergeben, mit zugeordnetem Auftragsbedarf von 15. Rest 10 würde durch unzugeordnete Bestellung bzw. Lager gedeckt werden - widerspricht Auftragsbilanz.
                                   -- Hier müssen bei Auftragsbilanz die Bezüge und Mengen eingetragen werden.
                    ) AS src
              GROUP BY aknr -- Summe pro Artikel
            ) AS auftragbilanz
       JOIN art ON art.ak_nr = auftragbilanz.aknr
      WHERE art.ak_ac LIKE inAkac
        AND auftragbilanz.qty < 0
      ;
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- Typ 30: Bedarf
  -- DROP FUNCTION IF EXISTS TEinkauf.bestellvorschlag__generate__from__Bedarf(varchar, date, varchar);
  CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlag__generate__from__Bedarf(
        inAkac varchar,
        inBdate date,
        inAuftg varchar
    )
    RETURNS TABLE(
        typ                         varchar,
        typindex                    integer,
        b_id                        integer,
        ak_nr                       varchar,
        ak_bez                      varchar,
        ak_tot                      numeric(12,4),
        ak_verfueg                  numeric(12,4),
        src_mgcode                  integer,
        standard_mgc_id             integer,
        b_date                      date,
        b_bestdat                   date,
        menge_uf1                   numeric,
        menge_los_uf1               numeric,
        bedarf_verursacher_list     tartikel.bedarf_verursacher[]
    )
    AS $$
    DECLARE
        -- ID mit letztem positiven Bestand aus Bedarfen (aka letzte Deckung). Danach beginnt Unterdeckung. Zur Ermittlung der nachfolgenden Bedarfsverursacher.
        b_id_last_positive_nbestand integer;

        -- Totale Bedarfsmenge zum Termin (Unterdeckung zum Termin). Zur Aufteilung auf gebunden Mengen der Bedarfsverursacher.
        bedarfsmenge_total          numeric(12,4);

        rec                         record;
    BEGIN

        inAkac  := coalesce( nullif( trim( inAkac ), '' ), '%' );
        inAuftg := coalesce( nullif( trim( inAuftg ), '' ), '%' );


        -- Ermittlung des Bedarfs und der zug. Bedarfsverursacher pro Artikel
        FOR
            -- Im LOOP direkte Zuweisung der OUT-Variablen
            typ,
            typindex,
            b_id,
            ak_nr,
            -- Hinweis: Überschriebene Artikelbezeichnung aus Auftrag?!
              -- nicht möglich: es müßte in den Bedarfen bereits gruppiert werden nach Artikelnummer und Bezeichnung
              -- Gruppierung wie in BANF anch ArtikelNr/Beizeichnung entfällt
              -- String_Agg wäre evtl eine Lösung?!
            ak_bez,
            ak_tot,
            ak_verfueg,
            src_mgcode,
            standard_mgc_id,
            menge_uf1,
            menge_los_uf1,
            b_id_last_positive_nbestand,
            bedarfsmenge_total,
            b_date,
            b_bestdat,
            bedarf_verursacher_list
        IN
            SELECT
              '30:Bedarf'::varchar  AS typ,
              30                    AS typindex,
              bedarf.b_id,
              art.ak_nr,
              art.ak_bez,
              art.ak_tot,
              art.ak_verfueg,
              null::integer         AS src_mgcode,
              tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(art.ak_nr),
              bedarf.b_nbestand     AS menge_uf1,
              -TArtikel.art__ak_los__qty_round(bedarf.b_aknr, -bedarf.b_nbestand) AS menge_los_uf1,
              coalesce( b_ids_last_positive_nbestand[1], 0 ),
              -bedarf.b_nbestand,
              -- zusätzliche Daten zur Ermittlung der Bedarfsverursacher-Liste.
              -- pro Artikel neu intialisieren
              null,
              null,
              null
            FROM (
                   -- ID-Aggregate aller Bedarfe pro Artikel im Zeitraum (performanteste Lösung). Resultiert in 1 datensatz pro Artikel.
                   SELECT
                           -- Alle Bedarf-IDs sortiert, um höchste anhand Sortierung zu ermitteln (hält alle Infos).
                           array_agg( bedarf.b_id ORDER BY bedarf.b_date DESC, bedarf.b_id DESC ) AS b_ids,

                           -- Alle Bedarf-IDs mit positiven Bestand sortiert, um letzten positiven Bestand zu ermitteln.
                           array_remove(
                               array_agg(
                                   CASE WHEN bedarf.b_nbestand >= 0
                                     THEN bedarf.b_id
                                   END
                                   ORDER BY bedarf.b_date DESC, bedarf.b_id DESC
                               ),
                               null
                           ) AS b_ids_last_positive_nbestand
                     FROM bedarf
                    WHERE coalesce(bedarf.b_bestdat, bedarf.b_date) <= inBdate -- Parameter (Bedarfsdatum)
                    GROUP BY bedarf.b_aknr -- pro Artikel
                  ) AS sub
              JOIN bedarf ON bedarf.b_id = b_ids[1] -- 1. Eintrag ist höchste anhand Termin und Sortierung zutreffene Bedarf-ID (hält alle Infos).
              JOIN art ON art.ak_nr = bedarf.b_aknr
            WHERE bedarf.b_nbestand < 0 -- Bei letztem Eintrag ist tatsächlich Bedarf vorhanden.
              AND art.ak_ac LIKE inAkac -- Parameter (spez. AC)
              AND EXISTS(SELECT true FROM bedarf AS bedarf_aus_auftg
                    WHERE bedarf_aus_auftg.b_aknr = bedarf.b_aknr
                          -- Parameter (Bedarf hat Bezug zu Auftragsnummer(n))
                      AND coalesce(bedarf_aus_auftg.b_auftg, '') LIKE inAuftg
                   )
        LOOP
            -- Bedarfsverursacher für aktuell ermittelten totalen Bedarf (zum Termin) durchgehen. #10007
            -- Zwischenzeitliche Unterdeckung mit nachfolgender Deckung (über 0) ist nicht relevant für aktuellen Bedarf (zum Termin).
            -- Es zählen für den aktuellen Bedarf nur die Bedarfsverursacher nach dem letzten positiven Bestand.
            -- Achtung! Kann auch Eintrag für Mindestbestand enthalten.
            FOR rec IN
                SELECT

                  -- BANF-Zusatzinfo für ABK-Materialposition (auftgi), siehe #15234
                  bap_id,
                  -- Bedarfsverursacher (Auftrag, BANF)
                  bv.b_ag_id, bv.b_bap_id,
                  -- Typ des Bedarfsverursachers
                  CASE
                      -- 1: Bedarf aus Auftrag
                      WHEN bv.b_ag_id  IS NOT NULL THEN 1
                      -- 2: Bedarf aus BANF
                      WHEN bv.b_bap_id IS NOT NULL THEN 2
                  END AS type_index,
                  -- Liefertermin an Lieferant (inkl Puffer von ABK her gesehen), vorgezogenes Bestelldatum
                  coalesce(bv.b_date_liefer, bv.b_date) AS b_date, bv.b_bestdat,
                  -- effektiver Bedarf durch Verursacher
                    -- Bsp: evtl. Überdeckung vor Abgang (80) + Abgang (-100) = effektiver Bedarf (-20)
                  -(greatest(bv.b_bestand, 0) + bv.b_zuab) AS bedarf_effektiv

                FROM bedarf AS bv
                  -- BANF enthält bloße Zusatzinfo (wenn keine Zusatzmenge) für ABK-Materialposition, siehe #15234
                  LEFT JOIN bestanfpos ON bap_ag_id = b_ag_id AND bap_menge = 0
                WHERE coalesce(bv.b_bestdat, bv.b_date) <= inBdate -- Parameter (Bedarfsdatum)
                  AND bv.b_aknr = ak_nr
                  AND bv.b_id > b_id_last_positive_nbestand -- Danach beginnt Unterdeckung für aktuell ermittelten totalen Bedarf (zum Termin).
                  AND bv.b_zuab < 0 -- nur Bedarfsverursacher
                ORDER BY bv.b_date, bv.b_id
            LOOP
                IF b_date IS NULL THEN

                    -- Das kleinste Bedarfsdatum, an dem der Bestand unter 0 fallen würde.
                    b_date    := rec.b_date;
                    -- Analog für vorgezogenes Bestelldatum (b_date - ak_bfr).
                    b_bestdat := rec.b_bestdat;

                END IF;


                -- Bedarfsverursacher-Liste (Array) für BV-Position aufbauen
                    -- Bedarfsverursacher vorhanden (Einträge mit Mindestbestand ausschließen)
                    IF rec.type_index IS NOT NULL THEN

                        -- zur Liste (Array) der Bedarfsverursacher hinzufügen.
                        bedarf_verursacher_list :=
                            bedarf_verursacher_list ||
                            (
                                -- Typ des Bedarfsverursachers
                                rec.type_index,
                                -- Bedarfsverursacher (Auftrag, BANF)
                                coalesce( rec.b_ag_id, rec.b_bap_id ),
                                -- effektiver Bedarf durch Verursacher
                                rec.bedarf_effektiv

                            )::tartikel.bedarf_verursacher
                        ;

                    END IF;

                    -- BANF-Zusatzinfo für ABK-Materialposition (auftgi) wird in Verursacherliste aufgenommen, siehe #15234
                      -- BANF enthält bloße Zusatzinfo (wenn keine Zusatzmenge) für ABK-Materialposition
                      -- Ziel: zusätzliche Texte und Lieferant aus Verursacherliste ermitteln
                      -- siehe bestellvorschlag__generate__txt__from__bedarf_verursacher_list
                    IF rec.bap_id IS NOT NULL THEN

                        -- zur Liste der Bedarfsverursacher hinzufügen.
                        bedarf_verursacher_list :=
                            bedarf_verursacher_list ||
                            -- BANF bzgl. Materialposition mit Menge 0
                            ( 2, rec.bap_id, 0 )::tartikel.bedarf_verursacher
                        ;

                    END IF;
                --

                bedarfsmenge_total := bedarfsmenge_total - rec.bedarf_effektiv;

                -- Totaler Bedarf aufgebraucht, dann raus und nächster Artikel.
                IF bedarfsmenge_total <= 0 THEN
                    EXIT;
                END IF;

            END LOOP;

            RETURN NEXT;

        END LOOP;

    END $$ LANGUAGE plpgsql;

  --

  -- Typ 2: Melde-Lag
  -- DROP FUNCTION IF EXISTS TEinkauf.bestellvorschlag__generate__from__melde__lag(varchar, varchar);
  CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlag__generate__from__melde__lag(
        inAkac varchar,
        inLgort varchar
    )
    RETURNS TABLE(
        typ                         varchar,
        typindex                    integer,
        b_id                        integer,
        ak_nr                       varchar,
        ak_bez                      varchar,
        ak_tot                      numeric(12,4),
        ak_verfueg                  numeric(12,4),
        src_mgcode                  integer,
        standard_mgc_id             integer,
        b_date                      date,
        b_bestdat                   date,
        menge_uf1                   numeric,
        menge_los_uf1               numeric,
        bedarf_verursacher_list     tartikel.bedarf_verursacher[]
    )
    AS $$
    BEGIN

        inAkac  := coalesce( nullif( trim( inAkac ), '' ), '%' );

        RETURN QUERY
        SELECT
               (lang_text(28215) || ': ' || lgoa_lgo_name)::varchar AS typ,
               20                  AS typindex,
               null::integer,
               lgoa_aknr,
               art.ak_bez,
               coalesce(sum_lg_anztot, 0),
               art.ak_verfueg,
               null::integer,
               tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(art.ak_nr),
               timediff_adddays(current_date, coalesce(art.ak_bfr, 0), true)::date,
               current_date,
               -( coalesce(lgoa_soll,0) + art.ak_res - art.ak_bes - coalesce(sum_lg_anztot, 0) ) AS menge_uf1,
               -TArtikel.art__ak_los__qty_round(art.ak_nr, (coalesce(lgoa_soll, 0) + art.ak_res - art.ak_bes - coalesce(sum_lg_anztot, 0))) AS menge_los_uf1,
               null::tartikel.bedarf_verursacher[]
          FROM lagartikelkonf
          JOIN art ON art.ak_nr = lgoa_aknr
          LEFT JOIN LATERAL (SELECT sum(lg_anztot) AS sum_lg_anztot FROM lag WHERE lg_ort = lgoa_lgo_name AND lg_aknr = lgoa_aknr) AS lagerbestand ON true
         WHERE lgoa_lgo_name LIKE inLgort
           AND art.ak_ac LIKE inAkac

               -- bestellmenge + lagernd kleiner meldebestand
           AND art.ak_bes - art.ak_res + coalesce(sum_lg_anztot, 0) < coalesce(lgoa_melde, 0)
        ;

    END $$ LANGUAGE plpgsql;
  --


  -- Typ 1: Melde-Art
  -- DROP FUNCTION IF EXISTS TEinkauf.bestellvorschlag__generate__from__melde__art(varchar);
  CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlag__generate__from__melde__art(
        inAkac varchar
    )
    RETURNS TABLE(
        typ                         varchar,
        typindex                    integer,
        b_id                        integer,
        ak_nr                       varchar,
        ak_bez                      varchar,
        ak_tot                      numeric(12,4),
        ak_verfueg                  numeric(12,4),
        src_mgcode                  integer,
        standard_mgc_id             integer,
        b_date                      date,
        b_bestdat                   date,
        menge_uf1                   numeric,
        menge_los_uf1               numeric,
        bedarf_verursacher_list     tartikel.bedarf_verursacher[]
    )
    AS $$
    BEGIN
        inAkac  := coalesce(nullif(trim(inAkac), ''), '%');

        RETURN QUERY
          SELECT lang_text(28214),
                 10                 AS typindex,
                 null::integer,
                 art.ak_nr,
                 art.ak_bez,
                 art.ak_tot,
                 art.ak_verfueg,
                 null::integer,
                 tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(art.ak_nr),
                 timediff_adddays(current_date, coalesce(art.ak_bfr, 0), true)::date,
                 current_date,
                 -- -(Sollmenge - Lagernd + Verkauft - Bestellt)
                 -(art.ak_soll - art.ak_tot + art.ak_res - art.ak_bes) AS menge_uf1,
                 -- Menge auf Los runden
                 -TArtikel.art__ak_los__qty_round(art.ak_nr, (art.ak_soll - art.ak_tot + art.ak_res - art.ak_bes)) AS menge_los_uf1,
                 null::tartikel.bedarf_verursacher[]
            FROM art
            WHERE
              NOT EXISTS (SELECT true FROM lagartikelkonf WHERE lgoa_aknr = art.ak_nr)
                  -- bestellmenge + lagernd kleiner meldebestand
              AND art.ak_bes + art.ak_tot - art.ak_res < art.ak_melde
                  -- ... wenn auch Meldebestand oder Sollbestand vorhanden ist
              AND coalesce(ak_melde, ak_soll, 0) > 0
              AND art.ak_ac LIKE inAkac
        ;
    END $$ LANGUAGE plpgsql;
  --
 --
 -- Bestellvorschlag: Verursacherliste / Bedarfsverursacher (ldsauftg)
  -- Zuständige Bearbeiter aggregiert anzeigen
  CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlagpos__ldsauftg__verursacherlist__erzeuger__agg(
        _bvp_id integer,
        OUT bvp_verursacher_ersteller_name varchar
        )
    AS $$

      SELECT
          string_agg(
              DISTINCT nameAufloesen(
                  coalesce(
                      ld_kontakt,
                      ldsdoki.insert_by,
                      ag_kontakt,
                      auftg.insert_by,
                      bestanfpos.insert_by
                  )
              ),
              ' ;'
          ) AS bvp_verursacher_ersteller_name
        FROM ldsauftg
        LEFT JOIN auftg ON ag_id = la_ag_id
        LEFT JOIN ldsdok ldsdoki ON ld_abk = ag_mainabk
        LEFT JOIN bestanfpos ON bap_id = la_bap_id

        -- COLAESCE: Bedarfsverursacher ist externer Auftrag direkt, dann gibt es kein ldsdok
        LEFT JOIN art ON ak_nr = coalesce( ld_aknr, ag_aknr, bap_aknr )
      WHERE la_bvp_id = _bvp_id
        AND (
                 la_bap_id IS NOT NULL
              OR la_ag_id IS NOT NULL
            )

    $$ LANGUAGE SQL;

  -- Bestellvorschlag auf Bedarfsverursacher einschränken https://redmine.prodat-sql.de/issues/13333
  CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlagpos__ldsauftg__verursacherlist__assign(
      _dest_bvp_id              integer,
      _Flags_Enum               varchar,
      VARIADIC _from_la_id_list integer[]
    ) RETURNS bool AS $$
    BEGIN

        -- Bestellvorschlagspos auf Auswahl einschränken (Mengen, Datum usw)
        UPDATE bestvorschlagpos SET
            bvp_menge        = ifthen(
                                  bvs_doLos,
                                  twawi.round_tolos( bvp_eklos, menge ),
                                  menge
                               ),
            bvp_bestdat      = date_larger( termin - ak_bfr, current_date ),
            bvp_ldatum       = date_larger( termin, current_date ),
            bvp_bedarf_list  = bedarf_verursacher
        FROM
            (
              SELECT
                  sum( coalesce(ag_stk, bap_menge) ) AS menge,

                  min(
                    coalesce( ag_kdatum, ag_ldatum, bap_termin, termweek_to_date( bap_termweek ), current_date )
                  ) AS termin,

                  -- array of tartikel.bedarf_verursacher
                  array_agg(
                      (
                        ifthen( ag_id IS NOT NULL, 1, 2),
                        ifthen( ag_id IS NOT NULL, ag_id, bap_id ),
                        coalesce( ag_stk, bap_menge )
                      )::tartikel.bedarf_verursacher
                      ORDER BY coalesce( ag_kdatum, ag_ldatum, bap_termin, termweek_to_date( bap_termweek ), current_date )
                  ) AS bedarf_verursacher

              FROM ldsauftg
              LEFT JOIN auftg      ON ag_id  = la_ag_id
              LEFT JOIN bestanfpos ON bap_id = la_bap_id
              WHERE
                    la_bvp_id = _dest_bvp_id
                AND la_id IN ( SELECT * FROM unnest( _from_la_id_list ) )
            ) AS ldsauftg, art, bestvorschlag
        WHERE
              bvp_id = _dest_bvp_id
          AND bvp_aknr = ak_nr
          AND bvs_nr = bvp_bvsnr
        ;

        -- lsdauftg : löschen der nicht mehr zugeordneten Einträge
        DELETE FROM ldsauftg
        WHERE
              la_bvp_id = _dest_bvp_id
          AND la_id NOT IN ( SELECT * FROM unnest( _from_la_id_list ) );

        RETURN FOUND;

    END $$ LANGUAGE plpgsql;
  --

  -- Bestellvorschlag : Bedarfsverursacher Separieren = Kopieren https://redmine.prodat-sql.de/issues/13368
  CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlagpos__ldsauftg__verursacherlist__separate(
       _src_bvp_id               integer,
       _Flags_Enum               varchar,
       VARIADIC _from_la_id_list integer[]
    )
    RETURNS integer AS $$
    DECLARE
       _i                   integer;
       rec_bvp              BestVorschlagPos;
       recjb_bvp_defaults   jsonb;
       recjb_bvp            jsonb;
       rec                  record;
       path                 text[];
    BEGIN

        -- Wenn BV-Pos nur 1 Verusacher hat, kann nicht separiert werden.
        _i := count(1) FROM ldsauftg WHERE la_bvp_id = _src_bvp_id;

        IF coalesce(_i, 0) <= 1 THEN
            RETURN null;
        END IF;


        -- 1. BVP-Pos - datensatz kopieren
        -- 1.1 Zu kopierenden record holen
        SELECT *
          FROM bestvorschlagpos
          INTO rec_bvp
         WHERE bvp_id = _src_bvp_id
        ;


        -- 1.2 =-Werte holen
        rec_bvp.dbrid     := nextval( 'db_id_seq' );
        rec_bvp.bvp_id    := nextval( 'bestvorschlagpos_bvp_id_seq' );
        rec_bvp.bvp_pos   := rec_bvp.bvp_pos + 1;

        -- Zurücksetzen falls die Ausgangsposition bereits ausgelöst wurde
        rec_bvp.bvp_einkauf_p_id  := null;
        rec_bvp.bvp_aArt_id       := null;
        rec_bvp.insert_date       := null;
        rec_bvp.insert_by         := null;
        rec_bvp.modified_date     := null;
        rec_bvp.modified_by       := null;
        rec_bvp.bvp_bedarf_list   := null;

        INSERT INTO bestvorschlagpos
        SELECT (rec_bvp).*;

        -- Verursacher entsprechend umhängen
        UPDATE ldsauftg
           SET la_bvp_id = rec_bvp.bvp_id
         WHERE la_bvp_id = _src_bvp_id
           AND la_id IN (
                  SELECT * FROM unnest(_from_la_id_list)
               )
        ;

        -- Assign beide Positionen mit ihren jetzigen Verursacherliste zum aktualisieren von Mengen, Termin usw.
        -- Neue Position einschränken auf die hier übergebene Liste
        PERFORM TEinkauf.bestellvorschlagpos__ldsauftg__verursacherlist__assign(
                    rec_bvp.bvp_id,
                    null,
                    VARIADIC _from_la_id_list
                )
        ;

        -- Bestehende Funktion einschränken auf das Übrige
        PERFORM TEinkauf.bestellvorschlagpos__ldsauftg__verursacherlist__assign(
                    _src_bvp_id,
                    null,
                    VARIADIC (
                        SELECT array_agg( la_id )
                        FROM ldsauftg
                        WHERE la_bvp_id = _src_bvp_id
                    )
                )
        ;


        RETURN rec_bvp.bvp_id;
    END $$ LANGUAGE plpgsql;
  --
 --

 -- FUNCTION teinkauf.bestvorschlag__check_chargenpflicht || Prüft anhand ldsauftg.la_ag_id die Bedarfsverursacher, ob diese chargenpflichig anhand ihrer Oberartikel sind, JM, #8207
  CREATE OR REPLACE FUNCTION teinkauf.bestvorschlag__check_chargenpflicht(
        agid integer,

        -- der Artikel selbst (z.B. das Rohmaterial) ist chargenpflichtig; true/false
        OUT bvs_selbst_chrg boolean,

        -- mein Kopfartikel (z.B. der Fertigungsartikel oder die Baugruppe) ist chargenpflichtig; true/false
        OUT bvs_kopf_chrg boolean,

        -- liefert das im bestellvorschlagpos-anzuzeigende Ergebnis; true = chargenpfl.; false = nicht chargenpfl.; null = abgeschwächt, nicht chargenpfl.
        OUT bvs_result boolean
    ) AS $$
    BEGIN

        -- agid => datensatz welcher Bedarf repräsentiert
        IF agid IS NULL THEN

            bvs_kopf_chrg    := false;
            bvs_selbst_chrg  := false;
            bvs_result       := false;
        ELSE

            bvs_kopf_chrg    := (EXISTS (SELECT true FROM auftg JOIN ldsdok ON ld_abk = ag_mainabk JOIN art ON ak_nr = ld_aknr WHERE ag_id = agid AND ak_chnrreq) OR
                                 EXISTS (SELECT true FROM auftg JOIN ldsdok ON ld_abk = ag_mainabk JOIN art ON ak_nr = ld_aknr JOIN artcod ON ak_ac = ac_n WHERE ag_id = agid AND (ac_charge_abk OR ac_charge_wen OR ac_charge_dat)));
            bvs_selbst_chrg  := (EXISTS (SELECT true FROM auftg JOIN art ON ak_nr = ag_aknr WHERE ag_id = agid AND ak_chnrreq) OR
                                 EXISTS (SELECT true FROM auftg JOIN art ON ak_nr = ag_aknr JOIN artcod ON ak_ac = ac_n WHERE ag_id = agid AND (ac_charge_abk OR ac_charge_wen OR ac_charge_dat)));
        END IF;

        -- Kopf erzwingt Charge
        bvs_result := bvs_kopf_chrg OR bvs_selbst_chrg;

   END $$ LANGUAGE plpgsql;
 --


 -- Neuer BV: holen spezifischer Daten aus dem zugehörigem vor BV
 CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlagpos__data_save__new(
      IN _bv_nr varchar
      )
      RETURNS bool
      AS $$
      BEGIN
            -- Falls wir in einem stehen und auf Neu erstellen (anstatt Aktualisieren) klicken auch löschen
            DELETE FROM bestvorschlagpos WHERE bvp_bvsnr = _bv_nr AND bvp_einkauf_p_id IS NULL;

            DROP TABLE IF EXISTS BestVorschlagPos_SaveDataForRefresh;
            -- oben wurde der evtl vorhandene aktuelle BV gelöscht. Somit bekommen wir die Daten aus dem letzten offenen BV des Artikels
            -- alle laufenden Positionen über alle BVs
            CREATE TEMP TABLE BestVorschlagPos_SaveDataForRefresh AS
            SELECT bestvorschlagpos.*,
                   -- Kennung Bestellvorschlag (Wöchentlich, ungebunden etc)
                   bvs_bvsk_id
              FROM bestvorschlagpos JOIN bestvorschlag ON bvs_nr = bvp_bvsnr
             WHERE NOT bvp_closed
                   -- Kennung unabhängig / ungebunden ausschliessen
               AND bvs_bvsk_id <> 1;

            RETURN found;

      END $$ LANGUAGE plpgsql;


 -- Aktualisieren BV: beibehalten der eigenen Daten
 CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlagpos__data_save__refresh(
      IN _bv_nr varchar
      )
      RETURNS bool
      AS $$
      BEGIN
            DROP TABLE IF EXISTS BestVorschlagPos_SaveDataForRefresh;

            -- Nur die laufenden Positionen des aktuellen BVs
            CREATE TEMP TABLE BestVorschlagPos_SaveDataForRefresh AS
            SELECT bestvorschlagpos.*,
                   -- siehe unten bvs_bvsk_id. null siehe RestoreData
                   null::int AS bvs_bvsk_id
              FROM bestvorschlagpos
             WHERE bvp_bvsnr = _bv_nr
                   -- bvs_bvsk_id : da wir nur im gleichen BV bleiben, ist die Kennung egal weil wir ja nur wir selbst sind
               AND NOT bvp_closed;

            -- Einkauf.BestVorschlag.NewBV_10_SaveData
            -- im Unterschied zum anderem SQL werden hier die Daten auch aus dem aktuellen BV erhalten. Das löschen erfolgt erst jetzt, in der TEMP TABLE sind wir auch selbst mit drin
            DELETE FROM bestvorschlagpos WHERE bvp_bvsnr = _bv_nr AND NOT bvp_closed;

            RETURN found;

      END $$ LANGUAGE plpgsql;



 -- Zurückholen der zwischengespeicherten Daten
 CREATE OR REPLACE FUNCTION TEinkauf.bestellvorschlagpos__data_save__restore(
      IN _bv_nr varchar
      )
      RETURNS bool
      AS $$
      BEGIN

            UPDATE bestvorschlagpos
               SET bvp_db_usename = usename,
                   bvp_ks = ks, bvp_konto = konto,
                   -- Wenn die Position angefragt wurde
                   bvp_aart_id = aart_id,
                   -- Lieferant
                   bvp_lkn = lkn,

                   bvp_status = -- wenn sich nichts geändert hat, dann wird der Status übernommen
                                CASE WHEN (bvp_bedarf_list = bedarf_list) AND bvp_menge = menge AND bvp_bestdat = bestdat AND bvp_ldatum = ldatum THEN
                                     status
                                ELSE
                                     -- Es gab einen Status, der aber aufgehoben wurde, da sich Bedarfe verändert haben
                                     IFTHEN(status IS NOT NULL, '?', null)
                                END,
                   --bvp_txt = txt, bvp_txt_rtf = txt_rtf, bvp_txtint = txtint, bvp_txtint_rtf = txtint_rtf,
                   bvp_txtint_orga = txto, bvp_txtint_orga_rtf = txtortf

                   -- Preissuche-Daten? (Abzuschläge, Rahmenverknüpfung etc)

              FROM (
                SELECT bvp_aknr, bvp_ak_bez, bvp_typindex,
                       bvp_db_usename AS usename,
                       bvp_ks         AS ks,
                       bvp_konto      AS konto,
                       bvp_aart_id    AS aart_id,
                       bvp_lkn        AS lkn,

                       -- Status bei Bedarfsgleichheit/Veränderung
                       bvp_status     AS status,
                       bvp_bedarf_list AS bedarf_list,
                       bvp_menge      AS menge,
                       bvp_bestdat    AS bestdat,
                       bvp_ldatum     AS ldatum,

                       /* diese Texte kommen aus Vorprozessen (Artikelstamm, BANF)
                       bvp_txt        AS txt, -- Notizen
                       bvp_txt_rtf    AS txt_rtf,
                       bvp_txtint     AS txtint, -- Notizen
                       bvp_txtint_rtf AS txtint_rtf,
                       */

                       bvp_txtint_orga     AS txto,
                       bvp_txtint_orga_rtf AS txtortf,

                       bvs_bvsk_id

                  FROM BestVorschlagPos_SaveDataForRefresh
                    -- aus der neuesten / letzten Position des Artikels im Bestellvorschlag
                 WHERE bvp_id IN (SELECT max(bvp_id)
                                    FROM BestVorschlagPos_SaveDataForRefresh
                                   WHERE NOT bvp_closed
                                   GROUP BY bvp_aknr, bvp_ak_bez, bvp_typindex, bvs_bvsk_id
                                  )
              ) AS t,
              bestvorschlag AS bvkopf_mine
             WHERE bvp_bvsnr = _bv_nr
               AND bvs_nr = bvp_bvsnr
               -- nur Positionen aktualisieren, die noch nicht bestellt/angefragt und nur anfrage wurden. Bestellte sind fix und werden nicht mehr angefasst.
               AND NOT bvp_closed
               AND t.bvp_aknr = bestvorschlagpos.bvp_aknr
               AND coalesce(t.bvp_ak_bez, '') = coalesce(bestvorschlagpos.bvp_ak_bez, '')
               AND t.bvp_typindex = bestvorschlagpos.bvp_typindex
                   -- Kennung wird, wenn wir nur Refresh machen auf null gesetzt um hier mit coalesce zu arbeiten. siehe Refresh_SaveData
               AND coalesce(t.bvs_bvsk_id, bvkopf_mine.bvs_bvsk_id) = bvkopf_mine.bvs_bvsk_id
            ;

            RETURN found;

      END $$ LANGUAGE plpgsql;

--
--
CREATE OR REPLACE FUNCTION teinkauf.ldsauftg__references__by__ld_id__agg(IN _ld_id integer, OUT la_bap_ids integer[], OUT bap_banrs varchar(150), OUT ag_ids integer[], OUT ag_nrs varchar(150)) RETURNS record
    AS $$
        SELECT array_agg(la_bap_id)                             AS la_bap_ids,
               string_agg(DISTINCT bap_banr, ';')::varchar(150) AS bap_banrs,
               array_agg(ag_id)                                 AS ag_ids,
               string_agg(DISTINCT ag_nr, ';')::varchar(150)    AS ag_nrs
          FROM ldsauftg
          LEFT JOIN bestanfpos ON bap_id = la_bap_id
          LEFT JOIN auftg      ON ag_id  = la_ag_id
         WHERE la_ld_id = _ld_id

    $$ LANGUAGE sql STABLE PARALLEL SAFE;



-- Liefermenge zu einer Bestellposition bis Datum oder in Zeitraum.
-- Variablenreihenfolge ist Erweiterung geschuldet.
CREATE OR REPLACE FUNCTION teinkauf.ld_stkl_fromwendat_todate(
    in_end_date date,
    in_ldid integer,
    in_begin_date date = null
    )
    RETURNS numeric(12,4)
    AS $$
    DECLARE
        r numeric(12,4);
    BEGIN
        r :=
          sum( w_zugang_uf1 )
          FROM wendat
          WHERE w_lds_id = in_ldid
            AND w_zug_dat::date BETWEEN coalesce( in_begin_date, w_zug_dat::date ) AND in_end_date
        ;
           RETURN coalesce( r, 0 );
    END $$ LANGUAGE plpgsql STABLE;
--


-- Auftragsbestätigungsnummer aus Schlagworten DMS setzen
CREATE OR REPLACE FUNCTION teinkauf.ldsdok__ld_abnr__from__dmskeywords__update(
    IN _ld_id     integer,
    IN _ld_abnr   varchar,
    IN _ld_abdat  date
    )
    RETURNS bool
    AS $$
    BEGIN
        UPDATE ldsdok
           SET ld_abnr  = coalesce( ld_abnr, _ld_abnr ),
               ld_abdat = coalesce( ld_abdat, _ld_abdat, current_date )
         WHERE ldsdok.ld_id = _ld_id
           AND (   ld_abnr  IS DISTINCT FROM coalesce( ld_abnr, _ld_abnr )
                OR ld_abdat IS DISTINCT FROM coalesce( ld_abdat, _ld_abdat, current_date )
                )
         ; -- pd_dokident

         RETURN found;
    END $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION teinkauf.anfrage__alief_angNr__from__dmskeywords__update(
    IN _alief_id      integer,
    IN _alief_angNr   varchar
    )
    RETURNS bool
    AS $$
    BEGIN
        UPDATE anflief
           SET alief_angNr = coalesce(alief_angNr, _alief_angNr)
         WHERE alief_id = _alief_id
           AND alief_angNr IS DISTINCT FROM coalesce(alief_angNr, _alief_angNr)
        ;

        RETURN found;
    END $$ LANGUAGE plpgsql;

--Bestellbezug über ld_id => Lagerumbuchungen
 CREATE OR REPLACE FUNCTION ldsdok__ld_id_ident( ldid integer ) RETURNS varchar AS $$

      SELECT twawi.ldsdok_genconcatnr( ldsdok ) || '=>' || ld_kn
        FROM ldsdok
       WHERE ld_id = ldid;

  $$ LANGUAGE SQL STABLE;

 -- Gesamte Bestellmenge pro AG einer ABK
 CREATE OR REPLACE FUNCTION ldsdok_aw_bestellt( inA2id integer ) RETURNS numeric AS $$
      SELECT coalesce( sum( ld_stk ), 0 )
        FROM ldsdok
       WHERE ld_a2_id = inA2id;
  $$ LANGUAGE SQL STABLE;
 --

 --Gesamte Liefermenge pro AG einer ABK
 CREATE OR REPLACE FUNCTION ldsdok_aw_rueck( inA2id integer ) RETURNS numeric AS $$
      SELECT coalesce( sum( ld_stkl ), 0 )
        FROM ldsdok
       WHERE ld_a2_id = inA2id;
  $$ LANGUAGE SQL STABLE;
 --

 CREATE OR REPLACE FUNCTION ldsdok_is_awbest( lddokunr integer ) RETURNS bool AS $$

    SELECT EXISTS(
          SELECT true
            FROM ldsdok
           WHERE ld_a2_id IS NOT NULL
             AND ld_dokunr = lddokunr
       )
    ;

   $$ LANGUAGE SQL STABLE;

--

-- Lieferanten anhand des Eingangsdatums bewerten
CREATE OR REPLACE FUNCTION teinkauf.lbwwe_get_termbw(
      warenein date,
      wunscht date,
      tofrueh integer = 0,
      tospaet integer = 0,
      OUT bez character varying,
      OUT punkte integer
  ) RETURNS record
  AS $function$
  DECLARE
      i integer;
  BEGIN

      -- tofrueh = positive Toleranz = negativer Wert
      -- tospaet = negative Toleranz = positiver Wert
      -- warenein = Wareneingang
      -- wunscht = Bestelldatum / Wunschtermin
      --  i = niedrigster definierter Abweichungswert aus der Terminbewetungstabelle(fruehester beschriebener Wert) - Toleranzwerte für verfrühte Lieferrung

      tofrueh := coalesce( tofrueh, 0 );
      tospaet := coalesce( tospaet, 0 );
      i := (SELECT min( lbwwe_abw ) FROM lbwwe )::integer - tofrueh;

      SELECT
         lbwwe_bez,
         lbwwe_punkte
      INTO bez, punkte
      FROM lbwwe
      WHERE
        CASE WHEN timediff(wunscht, warenein, true) < 0 AND  timediff(wunscht, warenein, true)::integer < i THEN -- Wenn die Lieferung so verfrüht ankommt, das der Wert ausserhald des in ldwwe definierten Wertebereichs liegt, dann denn niedrigsten Wert nehmen(schlechtest Bewertung bei zu früher Lieferung)
              lbwwe_abw = (SELECT MIN(lbwwe_abw) FROM lbwwe)
             WHEN timediff(wunscht, warenein, true) < 0 AND  timediff(wunscht, warenein, true)::integer > i AND timediff(wunscht, warenein, true) >= (-1*tofrueh) THEN -- Wenn die Lieferung so verfrüht ankommt, aber der Wert nicht ausserhalb des Wertebereichs liegt, aber innerhalb der Tolleranz, dann Wunschtermin
              lbwwe_punkte = (SELECT max(lbwwe_punkte) FROM lbwwe) --Wunschtermin = Minimale Abweichung = Maximale Punkte
             WHEN timediff(wunscht, warenein, true) < 0 AND  timediff(wunscht, warenein, true)::integer > i AND timediff(wunscht, warenein, true) < (-1*tofrueh) THEN -- Wenn die Lieferung so verfrüht ankommt, aber der Wert nicht ausserhalb des Wertebereichs liegt, aber ausserhalb der Tolleranz, dann Verspätung - Toleranz bewerten
               timediff(wunscht, warenein, true)::integer + tofrueh >= lbwwe_abw
             WHEN timediff(wunscht, warenein, true) >= 0 AND  timediff(wunscht, warenein, true)::integer > i AND timediff(wunscht, warenein, true) <= tospaet THEN -- Wenn die Lieferung so verspätet ankommt, aber der Wert nicht ausserhalb des Wertebereichs liegt, aber innerhalb der Tolleranz, dann Wunschtermin
              lbwwe_punkte = (SELECT max(lbwwe_punkte) FROM lbwwe) --Wunschtermin = Minimale Abweichung = Maximale Punkte
             ELSE
              timediff(wunscht,warenein, true)::integer - tospaet >= lbwwe_abw --UPDATE 26.01.2018 PHKO, vorher wurden Wochenenden von der Funktion nicht berücksichtigt, jetzt mit timediff
        END
      ORDER BY lbwwe_abw DESC
      LIMIT 1
      ;

      RETURN;

  END $function$ LANGUAGE plpgsql STABLE;
--

-- Ausgabe von Bedarfsverursachern (Aufträge, BANF) von Bestellung, Wareneingang oder Artikel.
-- Sortierung: 1. BANF vor Aufträgen; 2. direkte in Bestellung verlinkte vor bloß gebundenen; 3. nach Termin (vgl. Bedarfsberechnung); 4. nach Code Nr. Pos. des Bedarfsverursachers
CREATE OR REPLACE FUNCTION teinkauf.ldsdok__wendat__art__bedarfsverursacher_list(
    INOUT ldid        integer,          -- Bestellung führend vor Artikel.
    IN in__we_wen     integer = null,   -- Bei Angabe von WE ist Bestellung bzw. Artikel aus WE führend.
    IN in__ak_nr      varchar = null,   -- Ermittlung offener, nicht vollständig bestellter Bedarfsverursacher anhand Artikel
    IN with_descr     boolean = false,  -- Gibt normierte Beschreibung (Termin, Bedarfsverursacher, Zusatz) der Zeile mit aus.
    IN only_one_row   boolean = false,  -- Beschränkt die Ausgabe auf oberste Zeile.
    OUT agid          integer,
    OUT agparentabk   integer,          -- Bei Lagerzugängen auf Material, die direkt eine ABK zugeordnet sind - wird dieses Feld genutzt um direkt in den Lagerabgang ABK bezogen zu springen
    OUT bapid         integer,
    OUT description   varchar
    ) RETURNS SETOF record
    AS $$
    BEGIN
        IF in__we_wen IS NOT NULL THEN -- Bestellung und Artikel an WE ermitteln.
           IF EXISTS(SELECT true FROM wendat WHERE w_wen = in__we_wen) THEN -- Quellen nur überschreiben, wenn WE existiert.
               SELECT w_lds_id, w_aknr
                 INTO ldid,     in__ak_nr
                 FROM wendat
                WHERE w_wen = in__we_wen;
           END IF;
        END IF;

        IF ldid IS NULL AND in__ak_nr IS NULL THEN RETURN; END IF; -- keine Quellen, dann raus

        RETURN QUERY
          SELECT
                 ldid,
                 ag_id   AS agid,
                 abk     AS agparentabk,
                 bap_id  AS bapid,
                 CASE WHEN with_descr THEN
                     concat_ws(', '
                                , CASE WHEN direct_link THEN lang_text(21747)/*Fixe Materialzuordnung!*/ ELSE '' END || lang_text(755) || ': ' || to_char(termin, 'DD.MM.YY') -- Termin. 'Fix!' bei direktem Link in Bestellung.
                                , '(' || row_number() OVER pos_rank || '/' || count(*) OVER() || ') ' || -- mehrere per (1/n)
                                  CASE -- Nummer von Auftrag oder BANF
                                      WHEN ag_id IS NOT NULL THEN   format(lang_text(16719) || ': %s %s ' || lang_text(164) || ' %s', la_code, la_nummer, la_pos) -- AG: ... Pos. ...
                                      WHEN bap_id IS NOT NULL THEN  format(lang_text(16718) || ': %s '    || lang_text(164) || ' %s', la_nummer, la_pos)          -- BANF: ... Pos. ...
                                  END
                                , CASE -- Zusatz
                                      WHEN la_code = 'E' THEN       lang_text(494)    || ': ' || kunde          -- externe Aufträge mit Kunden
                                      WHEN la_code = 'I' THEN       lang_text(105)    || ': ' || abk            -- interne mit ABK
                                      WHEN bap_id IS NOT NULL THEN  lang_text(10390)  || ': ' || banf_ersteller -- Ersteller der BANF-Pos
                                  END
                              )
                 ELSE null END::varchar AS description
          FROM ( -- Komplex: Bitte Faltung arbeiten, Elemente sind alle sauber formatiert und kommentiert.
                 -- anhand Bestellung (ldid IS NOT NULL)
                 SELECT
                        ag_id,
                        bap_id,
                        coalesce(la_bap_id IS NOT null, la_ag_id IS NOT null) AS direct_link, -- BANF bzw. Auftrag ist direkt in Bestellung verlinkt.
                        CASE -- Termin, vgl. tartikel.bedarf__make_bedarf
                            WHEN ag_id IS NOT NULL THEN
                                 greatest(least( coalesce(ag_ldatum, ag_kdatum, termweek_to_date(ag_twa), current_date),
                                                 ag_aldatum
                                                 ),
                                          current_date
                                          )
                            WHEN bap_id IS NOT NULL THEN
                                 greatest(bap_termin, current_date)
                        END AS termin,
                        ag_astat AS la_code, coalesce(ag_nr, bap_banr) AS la_nummer, coalesce(ag_pos, bap_pos) AS la_pos,
                        ad_fa1 AS kunde,
                        ag_parentabk AS abk,
                        coalesce(nameAufloesen(bestanfpos.insert_by), lang_text(970)) AS banf_ersteller -- Fallback auf 'unbekannt'
                   FROM (
                         -- gebundene Bedarfsverursacher (ldsauftg)
                         SELECT
                                la_ag_id,
                                la_bap_id
                           FROM ldsauftg
                          WHERE la_ld_id = ldid
                            AND (la_ag_id IS NOT NULL OR la_bap_id IS NOT NULL)

                        ) AS bedarfsverursacher
                   JOIN ldsdok           ON ld_id = ldid
                   LEFT JOIN auftg       ON ag_id = la_ag_id   AND ag_astat IN ('E', 'I') AND NOT ag_done -- not ag/bap _done:  https://redmine.prodat-sql.de/issues/17797
                   LEFT JOIN bestanfpos  ON bap_id = la_bap_id AND NOT bap_prognose AND NOT bap_done
                   LEFT JOIN adk         ON ad_krz = ag_lkn
                  WHERE ldid IS NOT NULL
                 --

                 -- BANF des Artikels
                 -- Nur wenn keine Bestellung angg. ist, Bestellung ist führend (ldid IS NULL).
                 UNION
                 SELECT
                        null::integer AS ag_id,
                        bap_id,
                        false         AS direct_link,
                        greatest(bap_termin, current_date) AS termin, -- nach Termin, vgl. tartikel.bedarf__make_bedarf
                        null::varchar(1) AS la_code, bap_banr AS la_nummer, bap_pos AS la_pos,
                        null::varchar(100) AS kunde,
                        null::integer AS abk,
                        coalesce(nameAufloesen(bestanfpos.insert_by), lang_text(970)) AS banf_ersteller -- Fallback auf 'unbekannt'
                   FROM bestanfpos
                  WHERE ldid IS NULL
                    AND bap_aknr = in__ak_nr
                    AND NOT bap_prognose
                    AND NOT bap_done
                    AND coalesce(bap_menge_uf1, bap_menge) > bap_menge_best_uf1 -- noch nicht vollständig bestellt
                    --

                 -- Aufträge des Artikels
                 -- Nur wenn keine Bestellung angg. ist, Bestellung ist führend (ldid IS NULL).
                 UNION
                 SELECT
                        ag_id,
                        null::integer AS bap_id,
                        false         AS direct_link,
                        greatest(-- wenn in vergangenheit, dann heute, sonst wenn auslieferdatum das Kleinste, dann ag_aldatum
                                 least( coalesce(ag_ldatum, ag_kdatum, termweek_to_date(ag_twa), current_date),
                                        ag_aldatum
                                       ),
                                 current_date
                                 ) AS termin, -- nach Termin, vgl. tartikel.bedarf__make_bedarf
                        ag_astat AS la_code, ag_nr AS la_nummer, ag_pos AS la_pos,
                        ad_fa1 AS firma,
                        ag_parentabk AS abk,
                        null AS banf_ersteller
                   FROM auftg
                   LEFT JOIN adk ON ad_krz = ag_lkn
                  WHERE (    ldid IS NULL
                             -- wenn eine ld_id ABER keine gebundenen Verursacher (freie Bestellung) > dann auch dennoch anzeigen
                         OR (NOT EXISTS (SELECT true FROM ldsauftg WHERE la_ld_id = ldid))
                         )
                 AND ag_aknr = IfThen( ldid IS NULL, in__ak_nr, (SELECT ld_aknr FROM ldsdok WHERE ld_id = ldid) )
                 AND ag_astat IN ('E', 'I')
                 AND NOT ag_done
                 AND ag_stk_uf1 > ag_stkb -- noch nicht vollständig bestellt
                 --
                ) AS bedarfe
          WINDOW pos_rank AS ( -- für Window Function oben zur Ermittlung der Position
                              ORDER BY bap_id IS NOT NULL DESC, direct_link DESC, termin, la_code, la_nummer, la_pos -- ist identisch zur Sortierung des Querys
                              )
          ORDER BY
                bap_id IS NOT NULL DESC,  -- Banf bevorzugt ausgeben
                direct_link DESC,         -- direkt verlinkte BANF bzw. Auftrag als 1.
                termin,
                la_code, la_nummer, la_pos
          LIMIT CASE WHEN only_one_row THEN 1 ELSE null END
        ;

        RETURN;
    END $$ LANGUAGE plpgsql STABLE;
--

-- Funktion dient zur oberflächengesteuerten Übernahme von Mengenstaffeln von der Lieferantenanfrage zur Angebotsübernahme, ACJ, #13596
-- Parameter _est_aart_id identifiziert die zu kopierenden Mengenstaffel-datensätze anhand der zugeordneten Lieferantenanfrage
-- Parameter _est_aang_id das Angebot, welchem die kopierten datensätze zugeordnet werden
CREATE OR REPLACE FUNCTION teinkauf.epreisstaffel__copy_from__epreisstaffel(
      _est_aart_id integer,
      _est_aang_id integer
  ) RETURNS void AS $$
  BEGIN

      INSERT INTO epreisstaffel( est_mengevon, est_mengebis, est_aang_id )
      SELECT
          est_mengevon,
          est_mengebis,
          _est_aang_id
      FROM epreisstaffel
      WHERE est_aart_id = _est_aart_id;

  END $$ LANGUAGE plpgsql VOLATILE;
--

-- Funktion zählt die angefragten Lieferanten einer Lieferantenanfrage
-- Paramter _anf_nr identifiziert die betroffene Lieferantenanfrage
CREATE OR REPLACE FUNCTION teinkauf.anfrage__anflief__count__by__anf_nr( _anf_nr varchar )
  RETURNS int
  AS $$

     SELECT count(*)::int
       FROM anflief
      WHERE alief_anf_nr = _anf_nr

  $$ LANGUAGE sql;
--

-- Funktion zählt die mit Preisen erfassten Angebote zu einer Position einer Lieferantenanfrage
-- Paramter _aart_id identifiziert die betroffene Position der Lieferantenanfrage
CREATE OR REPLACE FUNCTION teinkauf.anfrage__anfart__anfangebot__count__by__aart_id( _aart_id int )
  RETURNS int
  AS $$

     SELECT count(*)::int
       FROM anfangebot
      WHERE aang_aart_id = _aart_id
        AND aang_preis IS NOT NULL;

  $$ LANGUAGE sql;
--

CREATE OR REPLACE FUNCTION teinkauf.anfangebot__update__aang_epreis__epreisstaffel(
      _aang_id    integer,    -- ID des Angebots
      _menge      numeric     -- erfasste Menge des Angebots
  ) RETURNS void AS $$
  DECLARE
      _est_ep numeric;
  BEGIN

      -- passt den Preis/ME eines Lieferantenangebots
      -- an den zugehörigen Staffelpreis an

      -- sucht den ersten Staffelpreis-Datensatz, welcher die
      -- übergebene Menge des übergebenen Angebots betrifft
      -- und liest den dortigen Preis aus
      _est_ep :=
          est_ep
          FROM epreisstaffel
          WHERE est_aang_id = _aang_id
            AND est_mengevon <= _menge
            AND (
                    _menge <= est_mengebis
                 OR est_mengebis IS null
            )
          ORDER BY est_id
          ASC LIMIT 1
      ;

      -- wenn ein passender Preis gefunden wurde,
      -- dann wird er in das Angebot übernehmen
      IF _est_ep IS NOT null THEN
          UPDATE anfangebot SET aang_ep = _est_ep WHERE aang_id = _aang_id;
      END IF;

  END $$ LANGUAGE plpgsql;
--

CREATE OR REPLACE FUNCTION teinkauf.anfangebot__aang_aLief_id__sum( _alief_id integer ) RETURNS numeric AS $$

      -- Ticket https://redmine.prodat-sql.de/issues/15954 - Ersetzung der Lagacy-Funktion getangebotsumme
      SELECT sum( aang_netto_basis_w ) FROM anfangebot WHERE aang_alief_id = _aLief_id;
  $$ LANGUAGE sql;
--

-- Beistellung

CREATE OR REPLACE FUNCTION TEinkauf.ldsdok__beistellung__abk__create(_ld_id integer) RETURNS integer
    AS $$
    DECLARE _ab_ix integer;
            _ldsdok record;
    BEGIN
        -- ABK anlegen für Beistellung mit korrektem Status zur Aufnahme der Beistellmaterialliste

        SELECT 'ldsdok'::varchar AS tname, dbrid,
               ld_abk, ld_stk,
               ENUM_GetValue(ld_stat, 'BL') AS beistellstat_existent
          INTO _ldsdok
          FROM ldsdok
         WHERE ld_id = _ld_id;

        -- Beistellung kann nicht mehrfach angelegt werden (die ABK welche die Materialliste aufnimmt)
        IF    NOT _ldsdok.ld_abk IS null
           OR     _ldsdok.beistellstat_existent
        THEN
            RAISE EXCEPTION '"ld_abk IS NOT null"??? OR ld_stat: beistellstat_existent > BL already exists???';
        END IF;

        IF EXISTS (SELECT true FROM abk WHERE ab_tablename = _ldsdok.tname AND ab_dbrid = _ldsdok.dbrid AND ab_stat = 'BL') THEN
            RAISE EXCEPTION 'abk "BL" already exists???';
        END IF;


        -- ABK anlegen, status setzen
        INSERT INTO abk
                    (ab_tablename, ab_dbrid,         ab_inplantaf, ab_st_uf1,      ab_stat)
             VALUES (_ldsdok.tname, _ldsdok.dbrid,   true,         _ldsdok.ld_stk, 'BL')
          RETURNING ab_ix INTO _ab_ix; -- Termine? Beachte F2-Fenster im Einkauf, welches die Termine für die einzelnen Materialien welche ausgewählt wurden setzt

        UPDATE ldsdok
           SET ld_abk = _ab_ix,
               ld_stat = ENUM_SetValue(ld_stat, 'BL')
         WHERE ld_id = _ld_id
           AND ld_abk IS null;

        RETURN _ab_ix;

    END $$ LANGUAGE plpgsql;

--Gibt alle Unterpositionen (ld_id) zurück
CREATE OR REPLACE FUNCTION teinkauf.ldsdok_do_ldsdoksubpos(_ldid integer) RETURNS SETOF integer AS $$
  DECLARE
    _hrec record;
    _rec record;
  BEGIN
    SELECT
      ld_code, ld_stat, ld_auftg, ld_pos
    FROM
      ldsdok
    WHERE
      ld_id = _ldid
    INTO _hrec;

    FOR _rec IN SELECT
                 ld_id
               FROM
                 ldsdok
               WHERE
                 ld_stat = _hrec.ld_stat
                 AND ld_code = _hrec.ld_code
                 AND ld_auftg = _hrec.ld_auftg
                 AND ld_hpos = _hrec.ld_pos
               ORDER BY
                 ld_pos
    LOOP
      RETURN NEXT _rec.ld_id;
    END LOOP;
    RETURN;
END $$ LANGUAGE plpgsql;

--Gibt die Kopfposition zurück
CREATE OR REPLACE FUNCTION teinkauf.ldsdok_get_ldsdokmainpos(_ldid integer) RETURNS integer AS $$
  DECLARE
    _rec record;
  BEGIN
    SELECT
      ld_code, ld_stat, ld_auftg, ld_hpos
    FROM
      ldsdok
    WHERE
      ld_id = _ldid
    INTO _rec;
    --
    IF _rec.ld_hpos IS NULL THEN --wenn es Kopf ist, sich selbst zurückgeben
      RETURN(_ldid);
    END IF;
    --
    RETURN (SELECT
                 ld_id
               FROM
                 ldsdok
               WHERE
                 ld_stat = _rec.ld_stat
                 AND ld_code = _rec.ld_code
                 AND ld_auftg = _rec.ld_auftg
                 AND ld_pos = _rec.ld_hpos
                 AND ld_hpos IS NULL);
END $$ LANGUAGE plpgsql;

--- #19731
CREATE OR REPLACE FUNCTION teinkauf.bestvorschlagpos__ldsauftg__zuschnitt__contains( IN _bvp_id integer ) RETURNS BOOLEAN AS $$
    SELECT exists( SELECT true
            FROM ldsauftg
               LEFT JOIN auftg ON ag_id = la_ag_id
               LEFT JOIN bestanfpos ON la_bap_id = bap_id
          WHERE la_bvp_id = _bvp_id
              AND ( ag_o6_stkz > 0 OR bap_o6_stkz > 0 ) );
 $$ LANGUAGE sql STABLE PARALLEL SAFE;
 --


-- #23077 Berechnung der Abzuschlagsbeträge eines Lieferantenangebots anhand der Prozentangabe
CREATE FUNCTION anfabzu__anfaz_betr__calculate( _anfang anfangebot ) RETURNS void AS $$

  UPDATE anfabzu
  SET anfaz_betr = _anfang.aang_preis * _anfang.aang_menge * anfaz_proz / 100.0
  WHERE
        anfaz_aang_id = _anfang.aang_id
    AND anfaz_proz IS NOT null
    AND anfaz_betr <> _anfang.aang_preis * _anfang.aang_menge * anfaz_proz / 100.0;

$$ LANGUAGE sql STRICT;
--
